package ru.yandex.chemodan.app.tcm.services;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.tcm.db.ConferenceMeta;
import ru.yandex.chemodan.app.tcm.db.ConferenceMetaDao;
import ru.yandex.chemodan.app.tcm.exceptions.ConferenceAlreadyExist;
import ru.yandex.chemodan.util.exception.BadRequestException;
import ru.yandex.commune.dynproperties.DynamicProperty;

/**
 * @author friendlyevil
 */
public class ConferenceMetaManager {
    private final DynamicProperty<Long> recentlyCreatedTimeSecond =
            new DynamicProperty<>("tcm-recently-conference-created-time[second]", 60L);
    private final ConferenceMetaDao conferenceMetaDao;

    public ConferenceMetaManager(ConferenceMetaDao conferenceMetaDao) {
        this.conferenceMetaDao = conferenceMetaDao;
    }

    public Option<ConferenceMeta> findConference(Option<String> shortUrlId, Option<String> conferenceId) {
        if (shortUrlId.isEmpty() && conferenceId.isEmpty()) {
            throw new BadRequestException("One of parameters short_url_id or conference_id must be not null");
        }

        if (shortUrlId.isPresent()) {
            return findConferenceByShortUrlId(shortUrlId.get());
        }

        return findConferenceByConferenceId(conferenceId.get());
    }

    public Option<ConferenceMeta> findConferenceByShortUrlId(String shortUrlId) {
        return conferenceMetaDao.findByShortUrlId(shortUrlId);
    }

    public Option<ConferenceMeta> findConferenceByConferenceId(String conferenceId) {
        return conferenceMetaDao.findByConferenceId(conferenceId);
    }

    public void createConferenceMeta(ConferenceMeta conferenceMeta) {
        ConferenceMeta conference = conferenceMetaDao.create(conferenceMeta);
        // Если дублирование conferenceId случилось недавно, то считаем что это ретрай и отдаем 200
        if (conference.getCreatedAt().plus(Duration.standardSeconds(recentlyCreatedTimeSecond.get())).isBeforeNow()) {
            throw new ConferenceAlreadyExist("Conference with conferenceId=" + conferenceMeta.getConferenceId() +
                    " already exist");
        }
    }

    public void updateConferenceMeta(ConferenceMeta conferenceMeta) {
        conferenceMetaDao.update(conferenceMeta);
    }
}
