package ru.yandex.chemodan.app.tcm.services;

import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.ToDoubleFunction;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.tcm.actions.CreateConferenceSettings;
import ru.yandex.chemodan.app.tcm.zk.ShardInfo;
import ru.yandex.chemodan.app.tcm.zk.ShardInfoZkRegistry;

/**
 * @author friendlyevil
 */
public class ShardResolver {
    private final ShardInfoZkRegistry shardInfoZkRegistry;

    public ShardResolver(ShardInfoZkRegistry shardInfoZkRegistry) {
        this.shardInfoZkRegistry = shardInfoZkRegistry;
    }

    public ShardInfo getShardInfoById(int shardId) {
        return shardInfoZkRegistry.get(shardId);
    }

    public ShardInfo getShardForNewConference(CreateConferenceSettings settings) {
        boolean yandex = settings.isStaffUid() || settings.isYandexTeamUid();
        if (yandex) {
            return getRandomYandexShard();
        }

        return getRandomExternalShard();
    }

    public ShardInfo getShardForUserInfo(CreateConferenceSettings settings) {
        return getShardForNewConference(settings);
    }

    public ShardInfo getFallbackShard() {
        return shardInfoZkRegistry.getAll().stream().filter(ShardInfo::isFallback).findAny()
                .orElseGet(this::getRandomShard);
    }

    private ShardInfo getRandomShard() {
        return getRandomFromWeighedShards(shardInfoZkRegistry.getAll().toList(), shardInfo -> 1.0);
    }

    private ShardInfo getRandomYandexShard() {
        ListF<ShardInfo> shardInfos =
                shardInfoZkRegistry.getAll().filter(ShardInfo::isSettleNewYandexConference).toList();
        return getRandomFromWeighedShards(shardInfos, ShardInfo::getYandexConferenceWeight);
    }

    private ShardInfo getRandomExternalShard() {
        ListF<ShardInfo> shardInfos =
                shardInfoZkRegistry.getAll().filter(ShardInfo::isSettleNewExternalConferences).toList();
        return getRandomFromWeighedShards(shardInfos, ShardInfo::getExternalConferenceWeight);
    }

    private ShardInfo getRandomFromWeighedShards(List<ShardInfo> shardInfos, ToDoubleFunction<ShardInfo> weight) {
        double total = shardInfos.stream().mapToDouble(weight).sum();
        double value = ThreadLocalRandom.current().nextDouble(total);
        double tmpSum = 0.0;

        for (ShardInfo shardInfo : shardInfos) {
            tmpSum += weight.applyAsDouble(shardInfo);
            if (tmpSum > value) {
                return shardInfo;
            }
        }

        throw new IllegalStateException("Shard for new conference not found");
    }
}
