package ru.yandex.chemodan.app.telemost.appmessages;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.appmessages.model.ErrorAppMessage;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class AppMessageProcessor {
    private static final Logger logger = LoggerFactory.getLogger(AppMessageProcessor.class);
    private final MapF<String, AppMessageHandler<?>> handlers;
    private final ObjectMapper objectMapper;

    public AppMessageProcessor(ListF<AppMessageHandler<?>> handlers, ObjectMapper objectMapper)
    {
        this.handlers = handlers.toMapMappingToKey(AppMessageHandler::getMessageType);
        Validate.unique(handlers.map(AppMessageHandler::getMessageType));
        this.objectMapper = objectMapper;
    }

    public TransportMessage processMessage(String roomId, String peerId, TransportMessage message) {
        Option<AppMessageHandler<?>> handlerO = handlers.getO(message.getType());
        if (!handlerO.isPresent()) {
            logger.error("Unable to find handler for message: {}, room_id: {}, peerId: {}, known messageTypes: {}",
                    roomId, peerId, message, handlers.keySet());
            return toJsonMessage(new ErrorAppMessage("unknown_type"));
        }

        AppMessageHandler<?> handler = handlerO.get();
        return toJsonMessage(processInternal(roomId, peerId, message, handler));
    }


    private <T extends AppMessage> AppMessage processInternal(
            String roomId, String peerId, TransportMessage request, AppMessageHandler<T> handler)
    {
        try {
            T appMessage = toAppMessage(request, handler.getMessageClass());
            return handler.processMessage(roomId, peerId, appMessage);
        } catch (MessageProcessingException e) {
            logger.error("Error during process message {}, conf_id = {}, peer_id = {}",
                    request, roomId, peerId, e);
            return new ErrorAppMessage(e.getCode());
        }
    }

    @VisibleForTesting
    <T extends AppMessage> T toAppMessage(TransportMessage message, Class<T> clazz) {
        try {
            return objectMapper.treeToValue(message.getPayload(), clazz);
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }
    }


    @VisibleForTesting
    TransportMessage toJsonMessage(AppMessage message) {
        return new TransportMessage(message.getMessageType(), objectMapper.valueToTree(message));
    }
}


