package ru.yandex.chemodan.app.telemost.appmessages;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.protobuf.ByteString;
import org.joda.time.Duration;

import ru.yandex.chemodan.app.telemost.appmessages.sender.MessageSender;
import ru.yandex.chemodan.app.telemost.room.proto.MediatorOuterClass;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.spring.Service;

public class AppMessageSender implements Service {
    private static final Logger logger = LoggerFactory.getLogger(AppMessageSender.class);

    private final MessageSender messageSender;
    private final ObjectMapper objectMapper;
    private final ThreadPoolExecutor executor;

    public AppMessageSender(MessageSender messageSender, ObjectMapper objectMapper, int idlePoolSize,
            int maxPoolSize, Duration keepAliveThreadTime)
    {
        this.messageSender = messageSender;
        this.objectMapper = objectMapper;
        this.executor = new ThreadPoolExecutor(idlePoolSize, maxPoolSize,
                keepAliveThreadTime.getStandardSeconds(), TimeUnit.SECONDS, new LinkedBlockingQueue<>());
    }

    @Override
    public void start() {
        executor.prestartAllCoreThreads();
    }

    @Override
    public void stop() {
        MoreExecutors.shutdownAndAwaitTermination(executor, 5, TimeUnit.SECONDS);
    }

    public void sendMessage(String roomId, AppMessage message, String peer) {
        MediatorOuterClass.SendAppMessageRequest request = MediatorOuterClass.SendAppMessageRequest.newBuilder()
                .setRoomId(roomId)
                .setSendToSpecific(MediatorOuterClass.SendAppMessageRequest.SendToSpecific.newBuilder()
                        .addPeers(peer)
                        .build())
                .setMessage(serializeMessage(message))
                .build();

        messageSender.sendAppMessage(roomId, request);
    }

    public CompletableFuture<?> sendMessageAsync(String roomId, AppMessage message, String peer) {
        return CompletableFuture.supplyAsync(
                MasterSlaveContextHolder.supplyWithStandardThreadLocal(
                        YandexCloudRequestIdHolder.supplyWithYcrid(
                                () -> {
                                    sendMessage(roomId, message, peer);
                                    return null;
                                })),
                executor
        );
    }

    public void sendMessageToAll(String roomId, AppMessage message,
            String... peerIdsToExclude)
    {
        MediatorOuterClass.SendAppMessageRequest request = MediatorOuterClass.SendAppMessageRequest.newBuilder()
                .setRoomId(roomId)
                .setSendToAll(buildSendAllDestination(peerIdsToExclude))
                .setMessage(serializeMessage(message))
                .build();

        messageSender.sendAppMessage(roomId, request);
    }

    public CompletableFuture<?> sendMessageToAllAsync(String roomId, AppMessage message, String... peerIdsToExclude) {
        return CompletableFuture.supplyAsync(
                MasterSlaveContextHolder.supplyWithStandardThreadLocal(
                        YandexCloudRequestIdHolder.supplyWithYcrid(
                                () -> {
                                    sendMessageToAll(roomId, message, peerIdsToExclude);
                                    return null;
                                })),
                executor
        );
    }

    @VisibleForTesting
    public boolean awaitAllProcessed(long time, TimeUnit timeUnit) {
        long start = System.nanoTime();
        long millisecondsToSleep = Math.min(50, timeUnit.toMillis(time));

        while ((System.nanoTime() - start) <= timeUnit.toNanos(time)) {
            if (isAllCompleted()) {
                return true;
            }
            try {
                Thread.sleep(millisecondsToSleep);
            } catch (InterruptedException e) {
                logger.warn("wait interrupted.", e);
                break;
            }
        }
        return isAllCompleted();
    }

    private boolean isAllCompleted() {
        return executor.getQueue().isEmpty() && executor.getActiveCount() == 0;
    }

    private ByteString serializeMessage(AppMessage message) {
        try {
            return ByteString.copyFrom(objectMapper.writeValueAsBytes(toJsonMessage(message)));
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private static MediatorOuterClass.SendAppMessageRequest.SendToAll.Builder buildSendAllDestination(
            String... peerIdsToExclude)
    {
        MediatorOuterClass.SendAppMessageRequest.SendToAll.Builder destination =
                MediatorOuterClass.SendAppMessageRequest.SendToAll.newBuilder();
        if (peerIdsToExclude != null && peerIdsToExclude.length > 0) {
            for (String peerId : peerIdsToExclude) {
                destination.addExcludePeers(peerId);
            }
        }

        return destination;
    }

    private TransportMessage toJsonMessage(AppMessage message) {
        return new TransportMessage(message.getMessageType(), objectMapper.valueToTree(message));
    }

}
