package ru.yandex.chemodan.app.telemost.appmessages.handlers;

import java.util.Map;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.telemost.appmessages.AppMessage;
import ru.yandex.chemodan.app.telemost.appmessages.AppMessageHandler;
import ru.yandex.chemodan.app.telemost.appmessages.model.ConferenceStateResponse;
import ru.yandex.chemodan.app.telemost.appmessages.model.GetPeerStateRequest;
import ru.yandex.chemodan.app.telemost.appmessages.model.GetPeersStateRequest;
import ru.yandex.chemodan.app.telemost.appmessages.model.PeerState;
import ru.yandex.chemodan.app.telemost.appmessages.model.PeersState;
import ru.yandex.chemodan.app.telemost.exceptions.ConferenceNotFoundTelemostException;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceUserDao;
import ru.yandex.chemodan.app.telemost.repository.dao.UserStateDtoDao;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceUserDto;
import ru.yandex.chemodan.app.telemost.repository.model.UserRole;
import ru.yandex.chemodan.app.telemost.repository.model.UserStateDto;
import ru.yandex.chemodan.app.telemost.services.ConferenceService;
import ru.yandex.chemodan.app.telemost.services.PeerPrefix;
import ru.yandex.chemodan.app.telemost.services.PeerStateService;
import ru.yandex.chemodan.app.telemost.services.model.Conference;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class GetPeersStateHandler implements AppMessageHandler<GetPeersStateRequest> {
    private static final Logger logger = LoggerFactory.getLogger(GetPeersStateHandler.class);

    private final UserStateDtoDao userStateDtoDao;
    private final ConferenceService conferenceService;
    private final ConferenceUserDao conferenceUserDao;
    private final PeerStateService peerStateService;

    private ListF<String> deleteBroadcasterPeerId(ListF<String> peerIds) {
        return peerIds.filter(x -> !x.startsWith(PeerPrefix.TRANSLATOR.getValue()));
    }

    @Override
    public AppMessage processMessage(String roomId, String peerId, GetPeersStateRequest message) {

        logger.info("Handle peers_state_get: rood_id={}, peer_id={}, message={}", roomId, peerId, message);

        Conference conference = conferenceService.findConferenceUnsafe(roomId)
                .getOrThrow(ConferenceNotFoundTelemostException::new);
        ListF<String> peerIds = Cf.x(message.getPeersState()).map(GetPeerStateRequest::getPeerId);

        peerIds = deleteBroadcasterPeerId(peerIds);

        MapF<String, ConferenceUserDto> members = conferenceUserDao.findByConferenceAndPeers(
                conference.getDbId(), peerIds);
        MapF<String, UserStateDto> states = addPassportDataIfRequired(
                userStateDtoDao.findStates(conference.getDbId(), peerIds), members);
        for (Tuple2<String, UserStateDto> entry : states.entries()) {
            UserRole userRole = members.getO(entry.get1()).map(ConferenceUserDto::getRole).orElse(UserRole.MEMBER);
            peerStateService.setRole(entry.get2(), userRole);
        }

        ListF<PeerState> responseStates = createResponse(message, states, roomId);

        return new PeersState(responseStates,
                message.getConferenceState() == null || conference.isDifferentVersion(message.getConferenceState().getVersion()) ?
                Option.of(new ConferenceStateResponse(conference.getConferenceState())) :
                Option.empty()
        );
    }

    private MapF<String, UserStateDto> addPassportDataIfRequired(MapF<String, UserStateDto> states, MapF<String, ConferenceUserDto> members) {
        if (states.values().stream().noneMatch(UserStateDto::isPassportDataRequired))
            return states;
        MapF<String, UserStateDto> result = Cf.hashMap();
        for (Map.Entry<String, UserStateDto> entry : states.entrySet()) {
            Option<PassportOrYaTeamUid> uid = members.getO(entry.getKey())
                    .map(ConferenceUserDto::getUid)
                    .map(PassportOrYaTeamUid::parseUid);
            result.put(entry.getKey(), peerStateService.storePassportDataIfRequired(entry.getValue(), uid));
        }
        return result;
    }

    private ListF<PeerState> createResponse(GetPeersStateRequest request, MapF<String, UserStateDto> states,
            String roomId)
    {
        ListF<PeerState> peerStates = Cf.arrayList();
        for (GetPeerStateRequest peer : request.getPeersState()) {
            Option<UserStateDto> stateO = states.getO(peer.getPeerId());
            if (!stateO.isPresent()) {
                logger.warn("requested state for unknown peer {} for conferenceId {}", peer.getPeerId(), roomId);
                continue;
            }
            UserStateDto state = stateO.get();

            if (peer.getVersion() != null && state.getVersion() == peer.getVersion()) {
                continue;
            }

            peerStates.add(new PeerState(peer.getPeerId(), state.getVersion(), state.getState()));
        }
        return peerStates;
    }

    @Override
    public Class<GetPeersStateRequest> getMessageClass() {
        return GetPeersStateRequest.class;
    }

    @Override
    public String getMessageType() {
        return GetPeersStateRequest.TYPE;
    }
}
