package ru.yandex.chemodan.app.telemost.appmessages.handlers;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.appmessages.AppMessage;
import ru.yandex.chemodan.app.telemost.appmessages.AppMessageHandler;
import ru.yandex.chemodan.app.telemost.appmessages.AppMessageSender;
import ru.yandex.chemodan.app.telemost.appmessages.model.PeerState;
import ru.yandex.chemodan.app.telemost.appmessages.model.PeersState;
import ru.yandex.chemodan.app.telemost.appmessages.model.SetPeerStateRequest;
import ru.yandex.chemodan.app.telemost.appmessages.model.SetPeerStateResponse;
import ru.yandex.chemodan.app.telemost.exceptions.ConferenceNotFoundTelemostException;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceUserDao;
import ru.yandex.chemodan.app.telemost.repository.dao.UserStateDtoDao;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceUserDto;
import ru.yandex.chemodan.app.telemost.repository.model.UserRole;
import ru.yandex.chemodan.app.telemost.repository.model.UserStateDto;
import ru.yandex.chemodan.app.telemost.services.ConferenceParticipantsService;
import ru.yandex.chemodan.app.telemost.services.ConferenceService;
import ru.yandex.chemodan.app.telemost.services.PeerPrefix;
import ru.yandex.chemodan.app.telemost.services.PeerStateService;
import ru.yandex.chemodan.app.telemost.services.model.Conference;
import ru.yandex.chemodan.app.telemost.services.model.ConferenceParticipant;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class SetPeerStateHandler implements AppMessageHandler<SetPeerStateRequest> {
    private static final Logger logger = LoggerFactory.getLogger(SetPeerStateHandler.class);

    private final UserStateDtoDao userStateDtoDao;
    private final ConferenceUserDao conferenceUserDao;
    private final ConferenceService conferenceService;
    private final ConferenceParticipantsService conferenceParticipantsService;
    private final PeerStateService peerStateService;
    private final AppMessageSender appMessageSender;

    @Override
    public AppMessage processMessage(String roomId, String peerId, SetPeerStateRequest message) {

        logger.info("Handle peers_state_set: rood_id={}, peer_id={}, message={}", roomId, peerId, message);

        Conference conference = conferenceService.findConferenceUnsafe(roomId)
                .getOrThrow(ConferenceNotFoundTelemostException::new);
        ConferenceParticipant participant = conferenceParticipantsService.findConferenceParticipant(conference, peerId);
        Option<UserStateDto> updatedState =
                userStateDtoDao.updatePeerState(participant.getDbUserId(), message.getStateContent())
                .map(state -> peerStateService.storePassportDataIfRequired(state, participant.getUid()));

        if (updatedState.isPresent()) {
            // Шлем стейт всем, кроме автора изменения, если стейт поменялся. Но не шлем, если изменился стейт
            // транслятора - он не должен рассылаться вообще.
            UserStateDto state = updatedState.get();
            UserRole userRole = participant.getUid()
                    .map(uid -> conferenceUserDao.findByConferenceAndUid(participant.getConferenceId(), uid))
                    .flatMapO(option -> option)
                    .map(ConferenceUserDto::getRole)
                    .orElse(UserRole.MEMBER);
            peerStateService.setRole(state, userRole);

            if (!peerId.startsWith(PeerPrefix.TRANSLATOR.getValue())) {
                appMessageSender.sendMessageToAllAsync(roomId,
                        new PeersState(Cf.list(new PeerState(peerId, state)), Option.empty()),
                        peerId);
            }
        } else {
            logger.info("State for peer {}@{} wasn't changed, it wouldn't be populated.", roomId, participant);
        }

        return SetPeerStateResponse.INSTANCE;
    }

    @Override
    public Class<SetPeerStateRequest> getMessageClass() {
        return SetPeerStateRequest.class;
    }

    @Override
    public String getMessageType() {
        return SetPeerStateRequest.TYPE;
    }
}
