package ru.yandex.chemodan.app.telemost.appmessages.sender;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.common.TelemostConstants;
import ru.yandex.chemodan.app.telemost.room.proto.MediatorOuterClass;
import ru.yandex.chemodan.app.telemost.room.proto.RoomGrpc;
import ru.yandex.chemodan.grpc.client.GrpcChannelBuilder;
import ru.yandex.chemodan.grpc.client.data.GrpcChannelConfiguration;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.lang.DefaultObject;

public class DynamicDestinationMessageSender implements MessageSender {
    private final RoomGrpc.RoomBlockingStub mediatorRoomGrpcService;
    private final LoadingCache<Endpoint, RoomGrpc.RoomBlockingStub> channelBuildersCache;

    private final DynamicProperty<Boolean> enableVerboseGrpcLogging = new DynamicProperty<>("telemost-grpc-logging-jicofo-verbose-enabled", Boolean.FALSE);

    public DynamicDestinationMessageSender(
            RoomGrpc.RoomBlockingStub mediatorRoomGrpcService,
            GrpcChannelBuilder jicofoChannelBuilder,
            Option<String> tlsServerName)
    {
        this.mediatorRoomGrpcService = mediatorRoomGrpcService;
        this.channelBuildersCache = CacheBuilder.newBuilder().maximumSize(500).build(
                new CacheLoader<Endpoint, RoomGrpc.RoomBlockingStub>() {
                    @Override
                    public RoomGrpc.RoomBlockingStub load(Endpoint e) {
                        return RoomGrpc
                                .newBlockingStub(jicofoChannelBuilder.buildChannel(
                                        GrpcChannelConfiguration.builder()
                                                .host(e.host)
                                                .port(e.port)
                                                .sslServerName(tlsServerName.getOrNull())
                                                .customInterceptors(Cf.list())
                                                .requestIdPrefix(TelemostConstants.GRPC_REQUEST_ID_PREFIX)
                                                .tracingMetadataKeyName(TelemostConstants.GRPC_MEDIATOR_REQUEST_TRACING_KEY)
                                                .enableLoggingVerboseMode(enableVerboseGrpcLogging::get)
                                                .build()));
                    }
                }
        );
    }

    @Override
    public void sendAppMessage(String roomId, MediatorOuterClass.SendAppMessageRequest request) {
        MediatorOuterClass.GetSFUEndpointResponse endpointResponse =
                mediatorRoomGrpcService.getSFUEndpoint(MediatorOuterClass.GetSFUEndpointRequest.newBuilder()
                        .setRoomId(roomId)
                        .build());

        Endpoint endpoint = new Endpoint(endpointResponse.getHost(), endpointResponse.getPort());
        channelBuildersCache.getUnchecked(endpoint).sendAppMessage(request);
    }

    private static class Endpoint extends DefaultObject {
        private final String host;
        private final int port;

        public Endpoint(String host, int port) {
            this.host = host;
            this.port = port;
        }
    }
}
