package ru.yandex.chemodan.app.telemost.calendar;

import java.io.IOException;
import java.net.URI;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.AllArgsConstructor;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.calendar.model.CalendarEvent;
import ru.yandex.chemodan.app.telemost.calendar.model.ResponseCalendarEvent;
import ru.yandex.chemodan.app.telemost.exceptions.ConferenceNotFoundTelemostException;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.commune.json.jackson.JodaTimeModule;
import ru.yandex.commune.json.jackson.bolts.BoltsModule;
import ru.yandex.inside.passport.tvm2.TvmHeaders;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.Abstract200ResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class CalendarClientImpl implements CalendarClient {
    private static final Logger logger = LoggerFactory.getLogger(CalendarClientImpl.class);

    private final HttpClient httpClient;
    private final URI calendarBaseCorpUrl;
    private final URI calendarBasePublicUrl;

    private static final String GET_EVENT = "get-event";

    private static final ObjectMapper mapper = new ObjectMapper()
            .registerModule(new BoltsModule())
            .registerModule(new JodaTimeModule())
            .setSerializationInclusion(JsonInclude.Include.NON_NULL);

    /**
     * GET EVENT - only for public calendar
     */

    @Override
    public CalendarEvent getEvent(String eventId, Option<PassportOrYaTeamUid> uid, Option<String> tvmUserTicketO) {
        UriBuilder uriBuilder = new UriBuilder(calendarBasePublicUrl).appendPath(GET_EVENT)
                .addParam("dateFormat", "zoned");
        if (uid.isPresent()) {
            uriBuilder = uriBuilder.addParam("uid", uid.get().withoutPrefix());
        }
        HttpPost request = new HttpPost(uriBuilder.addParam("eventId", eventId).build());
        tvmUserTicketO.forEach(userTicket -> request.addHeader(TvmHeaders.USER_TICKET, userTicket));

        loggingRequest(request);
        return ApacheHttpClientUtils.execute(request, httpClient, new Abstract200ResponseHandler<CalendarEvent>() {
            @Override
            protected CalendarEvent handle200Response(HttpResponse response) throws IOException {
                HttpEntity entity = response.getEntity();
                Check.notNull(entity);

                String strEntity = InputStreamSourceUtils.wrap(entity.getContent()).readText("utf-8");
                loggingResponse(strEntity);

                // Event was deleted
                if (strEntity.contains("event-not-found") && strEntity.contains("ru.yandex.calendar.frontend.web.cmd.run.CommandRunException")) {
                    throw new ConferenceNotFoundTelemostException();
                }

                ResponseCalendarEvent responseCalendarEvent = mapper.readValue(
                        strEntity,
                        ResponseCalendarEvent.class);
                return new CalendarEvent(responseCalendarEvent.getEventId(),
                        responseCalendarEvent.getEventCaption(),
                        responseCalendarEvent.getEventDescription(),
                        Instant.parse(responseCalendarEvent.getStartEvent()),
                        Instant.parse(responseCalendarEvent.getEndEvent()));
            }
        });
    }

    private void loggingRequest(HttpPost request) {
        StringBuilder headers = new StringBuilder();
        for (Header header: request.getAllHeaders()) {
            headers.append(header.toString()).append("; ");
        }
        logger.info("Calendar client request {} with headers {}", request.toString(), headers.toString());
    }

    private void loggingResponse(String entity) {
        logger.info("Calendar client response {}", entity);
    }

}
