package ru.yandex.chemodan.app.telemost.chat;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.UUID;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.AllArgsConstructor;
import lombok.NoArgsConstructor;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.telemost.chat.model.Chat;
import ru.yandex.chemodan.app.telemost.chat.model.ChatHistory;
import ru.yandex.chemodan.app.telemost.chat.model.ChatHistoryItem;
import ru.yandex.chemodan.app.telemost.chat.model.ChatRole;
import ru.yandex.chemodan.app.telemost.chat.model.ChatUser;
import ru.yandex.chemodan.app.telemost.services.ChatParamsService;
import ru.yandex.chemodan.app.telemost.services.model.ChatType;
import ru.yandex.chemodan.web.JacksonPojo;
import ru.yandex.commune.json.jackson.JodaTimeModule;
import ru.yandex.commune.json.jackson.bolts.BoltsModule;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.Abstract200ExtendedResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class ChatClientImpl implements ChatClient {

    private static final String NAMESPACE_HEADER = "X-Namespace";

    // For handle /meta_api
    private static final String CREATE_CHAT = "create_chat";
    private static final String GET_CHATS = "get_chats";
    private static final String UPDATE_MEMBERS = "update_members";
    private static final String REQUEST_USER = "request_user";
    private static final String GET_USER_INFO = "get_user_info";
    private static final String REMOVE_CHAT_MEMBERS = "remove_chat_members";

    // For handle /history
    private static final String HISTORY = "/history";

    // For handle /push
    private static final String PUSH = "push";

    private static final Logger logger = LoggerFactory.getLogger(ChatClientImpl.class);

    private static final ObjectMapper mapper = new ObjectMapper()
            .registerModule(new BoltsModule())
            .registerModule(new JodaTimeModule())
            .setSerializationInclusion(JsonInclude.Include.NON_NULL);

    private static final Long BATCH_LIMIT_MESSAGE_FOR_HISTORY = 1000L;

    private final HttpClient httpClient;
    private final URI chatBaseMetaUrl;
    private final URI chatFanoutUrl;
    private final String chatNamespace;

    private final ChatParamsService chatParamsService;

    /**
     * COMMON CLASSES
     */

    // Класс-обертка для вызова метода и его параметров.
    @AllArgsConstructor
    private static class WrapperRequestEntity extends DefaultObject implements JacksonPojo {
        @JsonProperty("method")
        private final String method;
        @JsonProperty("params")
        private final JsonNode params;

        protected static HttpEntity getEntity(String method, DefaultObject obj) {
            WrapperRequestEntity entity = new WrapperRequestEntity(method,
                    mapper.valueToTree(obj)
            );
            try {
                String payload = mapper.writeValueAsString(entity);
                logger.info("Meta-api: method={}, payload={}", method, payload);
                return new StringEntity(payload, ContentType.APPLICATION_JSON);
            } catch (JsonProcessingException e) {
                logger.info("Meta-api wrapper: method={}, error={}, message={}", method, e.toString(), e.getMessage());
                throw new UncheckedIOException(e);
            }
        }
    }

    // Класс-обертка для обработки результатов вызова метода.
    @AllArgsConstructor
    private static class WrapperResponseHandler<T> extends Abstract200ExtendedResponseHandler<T> {

        private final Function<JsonNode, T> nodeHandler;

        @Override
        protected T handle200Response(HttpResponse response) throws IOException {
            HttpEntity entity = response.getEntity();
            Check.notNull(entity);
            JsonNode jsonResponse = mapper.readValue(
                    InputStreamSourceUtils.wrap(entity.getContent()).readText("utf-8"),
                    JsonNode.class);
            return nodeHandler.apply(jsonResponse);
        }

    }

    /**
     * CREATE CHAT
     */

    @Override
    public Option<Chat> createChat(UUID chatId, ChatType chatType, String title, String description, Option<String> avatarId,
                                   ListF<UUID> admins, ListF<UUID> members) {
        HttpPost request = new HttpPost(chatBaseMetaUrl);
        request.setEntity(CreateChatRequestEntity.getEntity(chatId, title, description, avatarId, admins, members,
                chatParamsService.isPublic(chatType), chatNamespace));
        request.setHeader(NAMESPACE_HEADER, chatParamsService.getNamespaceId(chatType));
        logger.info("Call create chat before: base url={}, request={}, headers={}, entity={}, chatId={}, chatType={}",
                chatBaseMetaUrl, request, request.getAllHeaders(), request.getEntity(), chatId, chatType);
        Option<Chat> chatO = ApacheHttpClientUtils.execute(request, httpClient,
                new WrapperResponseHandler<>(x -> CreateChatResponseHandler.onHandle(x, chatType)));
        logger.info("Call create chat after: {}", chatO);
        return chatO;
    }

    @AllArgsConstructor
    private static class CreateChatRequestEntity extends DefaultObject implements JacksonPojo {
        @JsonProperty("name")
        private final String name;

        @JsonProperty("description")
        private final String description;

        @JsonProperty("avatar_id")
        private final Option<String> avatarId;

        @JsonProperty("chat_id")
        private final UUID chatId;

        @JsonProperty("roles")
        private final JsonNode roles;

        @JsonProperty("members")
        private final JsonNode members;

        @JsonProperty("public")
        private final boolean isPublic;

        @JsonProperty("namespace")
        private final String namespace;

        @JsonProperty("member_rights")
        private final JsonNode memberRights;

        protected static HttpEntity getEntity(UUID chatId, String title, String description, Option<String> avatarId,
                                              ListF<UUID> admins, ListF<UUID> members, boolean isPublic,
                                              String namespace) {
            CreateChatRequestEntity createChatRequestEntity = new CreateChatRequestEntity(
                    title, description, avatarId, chatId,
                    JsonNodeFactory.instance.objectNode().set("admin", mapper.valueToTree(admins.toArray())),
                    mapper.valueToTree(members.toArray()), isPublic, namespace,
                    JsonNodeFactory.instance.objectNode().set("pin_message", mapper.valueToTree(false)));
            logger.info("Create chat entity for chatId={}: {}", chatId, createChatRequestEntity);
            return WrapperRequestEntity.getEntity(CREATE_CHAT, createChatRequestEntity);
        }
    }

    @NoArgsConstructor
    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class CreateChatResponseHandler extends DefaultObject implements JacksonPojo {
        @JsonProperty("chat_id")
        private String chatPath;

        protected static Option<Chat> onHandle(JsonNode node, ChatType chatType) {
            if (!node.get("status").textValue().equals("ok")) return Option.empty();
            try {
                CreateChatResponseHandler chat = mapper.treeToValue(node.get("data"), CreateChatResponseHandler.class);
                logger.info("Create chat data for chatPath={}, chatType={}: {}", chat.chatPath, chatType, chat);
                return Option.of(new Chat(chat.chatPath,
                        chatType,
                        getMembers(node.get("data").get("members")),
                        Cf.list())
                );
            } catch (Exception e) {
                logger.info("Create chat: chatType={}, error={}, message={}", chatType, e.toString(), e.getMessage());
                throw ExceptionUtils.translate(e);
            }
        }

    }

    private static ListF<UUID> getMembers(JsonNode n) {
        Iterator<JsonNode> c = n.elements();
        ArrayList<UUID> members = new ArrayList<>();
        while (c.hasNext()) {
            members.add(UUID.fromString(c.next().asText()));
        }
        return Cf.x(members);
    }

    /**
     * GET CHATS
     */

    @Override
    public Option<Chat> getChats(UUID chatId, ChatType chatType) {
        HttpPost request = new HttpPost(chatBaseMetaUrl);
        request.setEntity(GetChatsRequestEntity.getEntity(chatParamsService.chatPath(chatId, chatType)));
        request.setHeader(NAMESPACE_HEADER, chatParamsService.getNamespaceId(chatType));
        return ApacheHttpClientUtils.execute(request, httpClient,
                new WrapperResponseHandler<>(x -> GetChatsResponseHandler.onHandle(x, chatType)));
    }

    @AllArgsConstructor
    private static class GetChatsRequestEntity extends DefaultObject implements JacksonPojo {
        @JsonProperty("chat_id")
        private final String chatPath;

        protected static HttpEntity getEntity(String chatPath) {
            return WrapperRequestEntity.getEntity(GET_CHATS, new GetChatsRequestEntity(chatPath));
        }
    }

    @NoArgsConstructor
    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class GetChatsResponseHandler extends DefaultObject implements JacksonPojo {
        protected static Option<Chat> onHandle(JsonNode node, ChatType chatType) {
            if (!node.get("status").textValue().equals("ok")) return Option.empty();
            try {
                Iterator<JsonNode> chats = node.get("data").get("chats").elements();
                if (chats.hasNext()) {
                    JsonNode chatNode = chats.next();
                    return Option.of(new Chat(chatNode.get("chat_id").asText(),
                            chatType,
                            getMembers(chatNode.get("members")),
                            getMembers(chatNode.get("subscribers")))
                    );
                }
                return Option.empty();
            } catch (Exception e) {
                logger.info("Get chats: error={}, message={}", e.toString(), e.getMessage());
                throw ExceptionUtils.translate(e);
            }
        }
    }

    /**
     * UPDATE MEMBERS
     */

    @Override
    public Option<Chat> updateMembers(UUID chatId, ChatType chatType, ListF<UUID> addMembers, ListF<UUID> removeMembers,
                                      ChatRole role) {
        HttpPost request = new HttpPost(chatBaseMetaUrl);
        request.setEntity(UpdateMembersRequestEntity.getEntity(chatParamsService.chatPath(chatId, chatType),
                addMembers, removeMembers, role.getValue()));
        request.setHeader(NAMESPACE_HEADER, chatParamsService.getNamespaceId(chatType));
        return ApacheHttpClientUtils.execute(request, httpClient,
                new WrapperResponseHandler<>(x -> UpdateMembersResponseHandler.onHandle(x, chatType)));
    }

    @AllArgsConstructor
    private static class UpdateMembersRequestEntity extends DefaultObject implements JacksonPojo {
        @JsonProperty("chat_id")
        private final String chatPath;

        @JsonProperty("role")
        private final String role;

        @JsonProperty("add")
        private final JsonNode forAdd;

        @JsonProperty("remove")
        private final JsonNode forRemove;

        @JsonProperty("return_members")
        private final boolean returnMembers = true;

        protected static HttpEntity getEntity(String chatPath, ListF<UUID> addMembers, ListF<UUID> removeMembers, String role) {
            return WrapperRequestEntity.getEntity(UPDATE_MEMBERS, new UpdateMembersRequestEntity(chatPath, role,
                    mapper.valueToTree(addMembers), mapper.valueToTree(removeMembers)));
        }
    }

    @NoArgsConstructor
    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class UpdateMembersResponseHandler extends DefaultObject implements JacksonPojo {
        protected static Option<Chat> onHandle(JsonNode node, ChatType chatType) {
            if (!node.get("status").textValue().equals("ok")) return Option.empty();
            try {
                return Option.of(new Chat(node.get("data").get("chat_id").asText(),
                        chatType,
                        getMembers(node.get("data").get("members")),
                        getMembers(node.get("data").get("subscribers")))
                );
            } catch (Exception e) {
                logger.info("Update members: error={}, message={}", e.toString(), e.getMessage());
                throw ExceptionUtils.translate(e);
            }
        }
    }

    /**
     * REQUEST USER
     */

    @Override
    public Option<UUID> getUser(String uid) {
        HttpPost request = new HttpPost(chatBaseMetaUrl);
        request.setEntity(RequestUserRequestEntity.getEntity(getClearUid(uid)));
        Option<ChatUser> chatUser = ApacheHttpClientUtils.execute(request, httpClient,
                new WrapperResponseHandler<>(RequestUserResponseHandler::onHandle));
        return chatUser.map(ChatUser::getUser);
    }

    @Override
    public MapF<String, UUID> getUsers(ListF<String> uids) {
        HashMap<String, UUID> users = new HashMap<>(uids.size());
        for (String uid : uids) {
            getUser(uid).map(user -> users.put(uid, user));
        }
        return Cf.x(users);
    }

    private String getClearUid(String uid) {
        if (uid.startsWith("yt:")) {
            return uid.substring(3);
        }
        return uid;
    }

    @AllArgsConstructor
    private static class RequestUserRequestEntity extends DefaultObject implements JacksonPojo {
        @JsonProperty("uid")
        private final String uid;

        protected static HttpEntity getEntity(String uid) {
            return WrapperRequestEntity.getEntity(REQUEST_USER, new RequestUserRequestEntity(uid));
        }
    }

    @NoArgsConstructor
    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class RequestUserResponseHandler extends DefaultObject implements JacksonPojo {
        @JsonProperty("uid")
        private String uid;

        @JsonProperty("guid")
        private UUID user;

        protected static Option<ChatUser> onHandle(JsonNode node) {
            if (!node.get("status").textValue().equals("ok")) return Option.empty();
            try {
                RequestUserResponseHandler user = mapper.treeToValue(node.get("data").get("user"), RequestUserResponseHandler.class);
                return Option.of(new ChatUser(user.uid, user.user));
            } catch (Exception e) {
                logger.info("Request user: error={}, message={}", e.toString(), e.getMessage());
                throw ExceptionUtils.translate(e);
            }
        }
    }

    /**
     * GET USER INFO
     */

    @Override
    public Option<String> getUserUid(UUID user) {
        HttpPost request = new HttpPost(chatBaseMetaUrl);
        request.setEntity(GetUserInfoRequestEntity.getEntity(user));
        return ApacheHttpClientUtils.execute(request, httpClient,
                new WrapperResponseHandler<>(GetUserInfoResponseHandler::onHandle));
    }

    @AllArgsConstructor
    private static class GetUserInfoRequestEntity extends DefaultObject implements JacksonPojo {
        @JsonProperty("guid")
        private final String user;

        protected static HttpEntity getEntity(UUID user) {
            return WrapperRequestEntity.getEntity(GET_USER_INFO, new GetUserInfoRequestEntity(user.toString()));
        }
    }

    @NoArgsConstructor
    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class GetUserInfoResponseHandler extends DefaultObject implements JacksonPojo {
        protected static Option<String> onHandle(JsonNode node) {
            if (!node.get("status").asText().equals("ok")) {
                return Option.empty();
            }
            // Field uid may be missing - no uid
            if (!node.get("data").has("uid")) {
                return Option.empty();
            }
            String uid = node.get("data").get("uid").asText();
            // Field uid can be zero or empty - no uid
            if (uid.isEmpty() || uid.equals("0")) {
                return Option.empty();
            }
            return Option.of(uid);
        }
    }

    /**
     * PUSH
     */

    @Override
    public boolean push(UUID chatId, ChatType chatType, UUID user, String message) {
        HttpPost request = new HttpPost(new UriBuilder(chatFanoutUrl).appendPath(PUSH).build());
        request.setEntity(PushRequestEntity.getEntity(chatParamsService.chatPath(chatId, chatType), user, message));
        request.setHeader(NAMESPACE_HEADER, chatParamsService.getNamespaceId(chatType));
        return ApacheHttpClientUtils.execute(request, httpClient,
                new WrapperResponseHandler<>(PushResponseHandler::onHandle));
    }

    @AllArgsConstructor
    private static class PushRequestEntity extends DefaultObject implements JacksonPojo {
        protected static HttpEntity getEntity(String chatId, UUID admin, String message) {
            ObjectNode genericMessage = JsonNodeFactory.instance.objectNode();
            genericMessage.put("MessageText", message);

            ObjectNode systemMessage = JsonNodeFactory.instance.objectNode();
            systemMessage.put("ChatId", chatId);
            systemMessage.put("PayloadId", UUID.randomUUID().toString());
            systemMessage.set("GenericMessage", genericMessage);

            ObjectNode clientMessage = JsonNodeFactory.instance.objectNode();
            clientMessage.set("SystemMessage", systemMessage);

            ObjectNode msg = JsonNodeFactory.instance.objectNode();
            msg.put("UserIp", "127.0.0.1");
            msg.put("Guid", admin.toString());
            msg.set("ClientMessage", clientMessage);
            try {
                // 12-байтовый заголовок требуется для указания версии протокола
                StringEntity en = new StringEntity(new String(new byte[]{5,0,0,0,0,0,0,0,0,0,0,0})
                        + mapper.writeValueAsString(msg), ContentType.APPLICATION_JSON);
                // Это особенность мессенджера - Content-Type должен быть жестко application/json (без charset)
                en.setContentType("application/json");
                return en;
            } catch (JsonProcessingException e) {
                logger.info("Push: error={}, message={}", e.toString(), e.getMessage());
                throw new UncheckedIOException(e);
            }
        }
    }

    @NoArgsConstructor
    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class PushResponseHandler extends DefaultObject implements JacksonPojo {
        protected static boolean onHandle(JsonNode node) {
            return node.has("Status") && node.get("Status").asText().equals("1");
        }
    }

    /**
     * HISTORY
     */

    @Override
    public Option<ChatHistory> history(UUID chatId, ChatType chatType, UUID user, Option<Long> offset) {
        HttpPost request = new HttpPost(new UriBuilder(chatFanoutUrl).appendPath(HISTORY).build());
        request.setEntity(HistoryRequestEntity.getEntity(chatParamsService.chatPath(chatId, chatType), user, offset));
        request.setHeader(NAMESPACE_HEADER, chatParamsService.getNamespaceId(chatType));
        return ApacheHttpClientUtils.execute(request, httpClient,
                new WrapperResponseHandler<>(HistoryResponseHandler::onHandle));
    }

    @AllArgsConstructor
    private static class HistoryRequestEntity extends DefaultObject implements JacksonPojo {
        @JsonProperty("ChatId")
        private final String chatPath;

        @JsonProperty("Limit")
        private final Long limit = BATCH_LIMIT_MESSAGE_FOR_HISTORY;

        @JsonProperty("Guid")
        private final UUID user;

        @JsonProperty("MaxTimestamp")
        private final Option<Long> maxTimestamp;

        protected static HttpEntity getEntity(String chatPath, UUID user, Option<Long> maxTimestamp) {
            try {
                StringEntity en = new StringEntity(mapper.writeValueAsString(new HistoryRequestEntity(chatPath, user, maxTimestamp)), ContentType.APPLICATION_JSON);
                // Это особенность мессенджера - Content-Type должен быть жестко application/json (без charset)
                en.setContentType("application/json");
                return en;
            } catch (JsonProcessingException e) {
                logger.info("History: error={}, message={}", e.toString(), e.getMessage());
                throw new UncheckedIOException(e);
            }
        }
    }

    @NoArgsConstructor
    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class HistoryResponseHandler extends DefaultObject implements JacksonPojo {
        protected static Option<ChatHistory> onHandle(JsonNode node) {
            ArrayList<ChatHistoryItem> chatHistoryItems = new ArrayList<>();
            JsonNode chats = node.withArray("Chats");
            if (chats.size() < 1) {
                return Option.empty();
            }
            JsonNode messages = chats.get(0).withArray("Messages");
            if (messages.size() == 0) {
                return Option.empty();
            }
            for (JsonNode message : messages) {
                JsonNode info = message.get("ServerMessage").get("ServerMessageInfo");
                JsonNode from = info.get("From");
                // Add only plain text messages
                if (message.get("ServerMessage").get("ClientMessage").has("Plain")) {
                    chatHistoryItems.add(new ChatHistoryItem(
                            info.get("Timestamp").asLong() / 1000,
                            from.get("DisplayName").asText(),
                            UUID.fromString(from.get("Guid").asText()),
                            message.get("ServerMessage").get("ClientMessage")));
                }
            }
            if (chatHistoryItems.isEmpty()) {
                return Option.empty();
            }
            return Option.of(new ChatHistory(Cf.x(chatHistoryItems),
                    chatHistoryItems.get(0).getTimestamp(),
                    chatHistoryItems.get(chatHistoryItems.size() - 1).getTimestamp()));
        }
    }

    @Override
    public boolean chatIsEmpty(UUID chatId, ChatType chatType, UUID user) {
        HttpPost request = new HttpPost(new UriBuilder(chatFanoutUrl).appendPath(HISTORY).build());
        request.setEntity(ChatIsEmptyRequestEntity.getEntity(chatParamsService.chatPath(chatId, chatType), user));
        request.setHeader(NAMESPACE_HEADER, chatParamsService.getNamespaceId(chatType));
        return ApacheHttpClientUtils.execute(request, httpClient,
                new WrapperResponseHandler<>(ChatIsEmptyResponseHandler::onHandle));
    }

    @AllArgsConstructor
    private static class ChatIsEmptyRequestEntity extends DefaultObject implements JacksonPojo {
        @JsonProperty("ChatId")
        private final String chatPath;

        @JsonProperty("Limit")
        private final Long limit = 3L;

        @JsonProperty("Guid")
        private final UUID user;

        protected static HttpEntity getEntity(String chatPath, UUID user) {
            try {
                StringEntity en = new StringEntity(mapper.writeValueAsString(new ChatIsEmptyRequestEntity(chatPath, user)), ContentType.APPLICATION_JSON);
                // Это особенность мессенджера - Content-Type должен быть жестко application/json (без charset)
                en.setContentType("application/json");
                return en;
            } catch (JsonProcessingException e) {
                logger.info("History: error={}, message={}", e.toString(), e.getMessage());
                throw new UncheckedIOException(e);
            }
        }
    }

    @NoArgsConstructor
    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class ChatIsEmptyResponseHandler extends DefaultObject implements JacksonPojo {
        protected static boolean onHandle(JsonNode node) {
            JsonNode chats = node.withArray("Chats");
            if (chats.size() < 1) {
                return true;
            }
            JsonNode messages = chats.get(0).withArray("Messages");
            for (JsonNode message: messages) {
                if (!message.get("ServerMessage").get("ClientMessage").has("SystemMessage")) {
                    return false;
                }
            }
            return true;
        }
    }

    /**
     * REMOVE CHAT MEMBERS
     */

    @Override
    public void removeChatMembers(UUID chatId, ChatType chatType) {
        HttpPost request = new HttpPost(chatBaseMetaUrl);
        request.setEntity(RemoveChatMembersRequestEntity.getEntity(chatParamsService.chatPath(chatId, chatType)));
        request.setHeader(NAMESPACE_HEADER, chatParamsService.getNamespaceId(chatType));
        ApacheHttpClientUtils.execute(request, httpClient,
                new WrapperResponseHandler<>(RemoveChatMembersResponseHandler::onHandle));
    }

    @AllArgsConstructor
    private static class RemoveChatMembersRequestEntity extends DefaultObject implements JacksonPojo {
        @JsonProperty("chat_id")
        private final String chatPath;

        protected static HttpEntity getEntity(String chatPath) {
            return WrapperRequestEntity.getEntity(REMOVE_CHAT_MEMBERS, new RemoveChatMembersRequestEntity(chatPath));
        }
    }

    @NoArgsConstructor
    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class RemoveChatMembersResponseHandler extends DefaultObject implements JacksonPojo {
        protected static Option<Chat> onHandle(JsonNode node) {
            return Option.empty();
        }
    }

}
