package ru.yandex.chemodan.app.telemost.config;

import java.util.List;
import java.util.Optional;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.appmessages.AppMessageHandler;
import ru.yandex.chemodan.app.telemost.appmessages.AppMessageProcessor;
import ru.yandex.chemodan.app.telemost.appmessages.AppMessageSender;
import ru.yandex.chemodan.app.telemost.appmessages.handlers.GetPeersStateHandler;
import ru.yandex.chemodan.app.telemost.appmessages.handlers.SetPeerStateHandler;
import ru.yandex.chemodan.app.telemost.appmessages.sender.DynamicDestinationMessageSender;
import ru.yandex.chemodan.app.telemost.appmessages.sender.MessageSender;
import ru.yandex.chemodan.app.telemost.appmessages.sender.MessageSenderImpl;
import ru.yandex.chemodan.app.telemost.appmessages.sender.MultipleDestinationMessageSender;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceUserDao;
import ru.yandex.chemodan.app.telemost.repository.dao.UserStateDtoDao;
import ru.yandex.chemodan.app.telemost.room.proto.RoomGrpc;
import ru.yandex.chemodan.app.telemost.services.ConferenceParticipantsService;
import ru.yandex.chemodan.app.telemost.services.ConferenceService;
import ru.yandex.chemodan.app.telemost.services.PeerStateService;
import ru.yandex.chemodan.grpc.client.GrpcChannelBuilder;
import ru.yandex.chemodan.web.JacksonContextConfiguration;
import ru.yandex.inside.passport.tvm2.Tvm2;

@Configuration
@Import({TelemostServicesContextConfiguration.class,
        JacksonContextConfiguration.class,
        TelemostRoomContextConfiguration.class,
})
public class TelemostAppMessageConfiguration {

    @Bean
    public GetPeersStateHandler getPeersStateHandler(UserStateDtoDao userStateDtoDao,
                                                     ConferenceService conferenceService,
                                                     ConferenceUserDao conferenceUserDao,
                                                     PeerStateService peerStateService) {
        return new GetPeersStateHandler(userStateDtoDao, conferenceService, conferenceUserDao, peerStateService);
    }

    @Bean
    public SetPeerStateHandler setPeerStateHandler(UserStateDtoDao userStateDtoDao, AppMessageSender appMessageSender,
                                                   ConferenceService conferenceService,
                                                   ConferenceParticipantsService conferenceParticipantsService,
                                                   PeerStateService peerStateService,
                                                   ConferenceUserDao conferenceUserDao) {
        return new SetPeerStateHandler(userStateDtoDao, conferenceUserDao, conferenceService, conferenceParticipantsService,
                peerStateService, appMessageSender);
    }

    @Bean
    public AppMessageProcessor appMessageProcessor(List<AppMessageHandler<?>> handlers, ObjectMapper objectMapper) {
        return new AppMessageProcessor(Cf.x(handlers), objectMapper);
    }

    @Bean
    public AppMessageSender appMessageSender(MessageSender messageSender, ObjectMapper objectMapper,
            @Value("${telemost.app-message-sender.async.pool.size.min}") int asyncSendPoolMinSize,
            @Value("${telemost.app-message-sender.async.pool.size.max}") int asyncSendPoolMaxSize,
            @Value("${telemost.app-message-sender.async.pool.thread.keep-alive-time}") Duration keepAliveThreadTime
    )
    {
        return new AppMessageSender(messageSender, objectMapper, asyncSendPoolMinSize, asyncSendPoolMaxSize,
                keepAliveThreadTime);
    }

    @Bean
    public MessageSender messageSender(MessageSender jicofoMessageSender, MessageSender mediatorMessageSender) {
        return new MultipleDestinationMessageSender(Cf.list(mediatorMessageSender, jicofoMessageSender));
    }

    @Bean
    public MessageSender mediatorMessageSender(RoomGrpc.RoomBlockingStub roomBlockingStub) {
        return new MessageSenderImpl(roomBlockingStub);
    }

    @Bean
    public MessageSender jicofoMessageSender(RoomGrpc.RoomBlockingStub roomBlockingStub,
            GrpcChannelBuilder jicofoGrpcChannelBuilder,
            @Value("${telemost.jicfo.tls.hostname}") Optional<String> jicofoTlsServerName)
    {
        return new DynamicDestinationMessageSender(roomBlockingStub, jicofoGrpcChannelBuilder,
                Option.x(jicofoTlsServerName));
    }

    @Bean
    public GrpcChannelBuilder jicofoGrpcChannelBuilder(Tvm2 tvm2,
            @Value("${telemost.jicofo.tvm.client-id}") Integer jicofoTvmClientId)
    {
        return new GrpcChannelBuilder(jicofoTvmClientId, Option.ofNullable(tvm2));
    }
}
