package ru.yandex.chemodan.app.telemost.config;

import io.grpc.ManagedChannel;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.common.TelemostConstants;
import ru.yandex.chemodan.app.telemost.logging.TelemostAuditLog;
import ru.yandex.chemodan.app.telemost.room.RoomBlockingManager;
import ru.yandex.chemodan.app.telemost.room.RoomManager;
import ru.yandex.chemodan.app.telemost.room.RoomSettingsHolder;
import ru.yandex.chemodan.app.telemost.room.proto.RoomGrpc;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsContextConfigutation;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.chemodan.grpc.client.DeadlineInterceptor;
import ru.yandex.chemodan.grpc.client.GrpcChannelBuilder;
import ru.yandex.chemodan.grpc.client.data.GrpcChannelConfiguration;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.tvm2.Tvm2;
import ru.yandex.inside.passport.tvm2.web.Tvm2BaseContextConfiguration;

@Configuration
@Import({Tvm2BaseContextConfiguration.class, ExperimentsContextConfigutation.class})
public class TelemostRoomContextConfiguration {

    @Value("${telemost.mediator.host}")
    private String mediatorHost;

    @Value("${telemost.mediator.port}")
    private int mediatorGrpcPort;

    @Value("${telemost.mediator.tvm.client-id}")
    private int mediatorTvmClientId;

    @Value("${telemost.grpc.client.deadline.ms}")
    private long deadlineMs;

    private final DynamicProperty<Boolean> enableVerboseGrpcLogging = new DynamicProperty<>("telemost-grpc-logging" +
            "-room-verbose-enabled", Boolean.FALSE);


    @Bean
    public TelemostAuditLog telemostAuditLog() {
        return new TelemostAuditLog();
    }

    @Bean
    public RoomSettingsHolder roomSettingsHolder(@Value("${telemost.self.app_message.endpoint}") String appServiceEndpoint) {
        return new RoomSettingsHolder(appServiceEndpoint);
    }

    @Bean
    public GrpcChannelBuilder mediatorGrpcChannelBuilder(Tvm2 tvm2) {
        return new GrpcChannelBuilder(mediatorTvmClientId, Option.ofNullable(tvm2));
    }

    @Bean
    public ManagedChannel mediatorChannel(GrpcChannelBuilder mediatorGrpcChannelBuilder) {
        return mediatorGrpcChannelBuilder.buildChannel(
                GrpcChannelConfiguration.builder()
                        .host(mediatorHost)
                        .port(mediatorGrpcPort)
                        .customInterceptors(Cf.list(new DeadlineInterceptor(deadlineMs)))
                        .requestIdPrefix(TelemostConstants.GRPC_REQUEST_ID_PREFIX)
                        .tracingMetadataKeyName(TelemostConstants.GRPC_MEDIATOR_REQUEST_TRACING_KEY)
                        .enableLoggingVerboseMode(enableVerboseGrpcLogging::get)
                        .build());
    }

    @Bean
    public RoomGrpc.RoomBlockingStub roomBlockingStub(ManagedChannel mediatorChannel) {
        return RoomGrpc.newBlockingStub(mediatorChannel);
    }

    @Bean
    public RoomGrpc.RoomStub roomStub(ManagedChannel mediatorChannel) {
        return RoomGrpc.newStub(mediatorChannel);
    }

    @Bean
    public RoomManager roomBlockingManager(RoomGrpc.RoomBlockingStub roomBlockingStub,
                                           RoomSettingsHolder roomSettingsHolder,
                                           ExperimentsManager experimentsManager, TelemostAuditLog telemostAuditLog) {
        return new RoomBlockingManager(roomBlockingStub, roomSettingsHolder, experimentsManager, telemostAuditLog);
    }

}
