package ru.yandex.chemodan.app.telemost.config;

import java.net.URI;
import java.util.List;
import java.util.UUID;
import java.util.function.Supplier;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.chemodan.app.telemost.appmessages.AppMessageSender;
import ru.yandex.chemodan.app.telemost.calendar.CalendarClient;
import ru.yandex.chemodan.app.telemost.calendar.CalendarClientImpl;
import ru.yandex.chemodan.app.telemost.chat.ChatClient;
import ru.yandex.chemodan.app.telemost.chat.ChatClientImpl;
import ru.yandex.chemodan.app.telemost.common.ShardPropertiesZkRegistry;
import ru.yandex.chemodan.app.telemost.orchestrator.TranslatorsOrchestrator;
import ru.yandex.chemodan.app.telemost.orchestrator.TranslatorsOrchestratorImpl;
import ru.yandex.chemodan.app.telemost.repository.dao.BroadcastDao;
import ru.yandex.chemodan.app.telemost.repository.dao.ClientConfigurationDao;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceDtoDao;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferencePeerDao;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceStateDao;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceUserDao;
import ru.yandex.chemodan.app.telemost.repository.dao.MediaSessionDao;
import ru.yandex.chemodan.app.telemost.repository.dao.StreamDao;
import ru.yandex.chemodan.app.telemost.repository.dao.UserDao;
import ru.yandex.chemodan.app.telemost.repository.dao.UserStateDtoDao;
import ru.yandex.chemodan.app.telemost.repository.dao.UserTokenDtoDao;
import ru.yandex.chemodan.app.telemost.repository.model.ApiVersion;
import ru.yandex.chemodan.app.telemost.room.RoomManager;
import ru.yandex.chemodan.app.telemost.room.proto.RoomGrpc;
import ru.yandex.chemodan.app.telemost.services.BroadcastService;
import ru.yandex.chemodan.app.telemost.services.BroadcastUriService;
import ru.yandex.chemodan.app.telemost.services.CalendarService;
import ru.yandex.chemodan.app.telemost.services.ChatParamsService;
import ru.yandex.chemodan.app.telemost.services.ChatService;
import ru.yandex.chemodan.app.telemost.services.ClientConfigurationService;
import ru.yandex.chemodan.app.telemost.services.CommandService;
import ru.yandex.chemodan.app.telemost.services.ConferenceCreator;
import ru.yandex.chemodan.app.telemost.services.ConferenceManagerClient;
import ru.yandex.chemodan.app.telemost.services.ConferenceParticipantsService;
import ru.yandex.chemodan.app.telemost.services.ConferencePeerService;
import ru.yandex.chemodan.app.telemost.services.ConferenceService;
import ru.yandex.chemodan.app.telemost.services.ConferenceUriService;
import ru.yandex.chemodan.app.telemost.services.DefaultConferenceCreator;
import ru.yandex.chemodan.app.telemost.services.LimitsService;
import ru.yandex.chemodan.app.telemost.services.OverloadService;
import ru.yandex.chemodan.app.telemost.services.ParticipantIdGenerator;
import ru.yandex.chemodan.app.telemost.services.PeerStateService;
import ru.yandex.chemodan.app.telemost.services.PropertyManager;
import ru.yandex.chemodan.app.telemost.services.PropertyManagerImpl;
import ru.yandex.chemodan.app.telemost.services.RoomService;
import ru.yandex.chemodan.app.telemost.services.StaffService;
import ru.yandex.chemodan.app.telemost.services.StreamService;
import ru.yandex.chemodan.app.telemost.services.UgcLiveService;
import ru.yandex.chemodan.app.telemost.services.UserService;
import ru.yandex.chemodan.app.telemost.services.UserTokenService;
import ru.yandex.chemodan.app.telemost.services.WebsocketUriService;
import ru.yandex.chemodan.app.telemost.services.YaTeamConferenceCreator;
import ru.yandex.chemodan.app.telemost.translator.TranslatorClient;
import ru.yandex.chemodan.app.telemost.translator.TranslatorClientImpl;
import ru.yandex.chemodan.app.telemost.ugcLive.UgcLiveClient;
import ru.yandex.chemodan.app.telemost.ugcLive.UgcLiveClientImpl;
import ru.yandex.chemodan.app.telemost.ugcLive.UgcLiveStreamPublisher;
import ru.yandex.chemodan.app.telemost.ugcLive.UgcLiveStreamPublisherImpl;
import ru.yandex.chemodan.app.telemost.util.UUIDUtils;
import ru.yandex.chemodan.blackbox.ProdBlackboxContextConfiguration;
import ru.yandex.chemodan.boot.value.OverridableValuePrefix;
import ru.yandex.chemodan.orchestrator.OrchestratorClient;
import ru.yandex.chemodan.orchestrator.OrchestratorClientContextConfiguration;
import ru.yandex.chemodan.tvm2.OAuthUserTicketSupplier;
import ru.yandex.chemodan.tvm2.UserTicketSupplier;
import ru.yandex.chemodan.util.ZkUtils;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;
import ru.yandex.chemodan.web.JacksonContextConfiguration;
import ru.yandex.chemodan.xiva.BasicXivaClient;
import ru.yandex.chemodan.zk.registries.staff.YandexStaffUserRegistry;
import ru.yandex.chemodan.zk.registries.staff.YandexStaffUserRegistryContextConfiguration;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.client.ZkManager;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.BlackboxType;
import ru.yandex.inside.passport.tvm2.Tvm2;
import ru.yandex.inside.tanker.TankerClient;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.parse.CommaSeparated;
import ru.yandex.misc.random.Random2;

@Configuration
@Import({
        TelemostRepositoryContextConfiguration.class,
        JacksonContextConfiguration.class,
        TelemostRoomContextConfiguration.class,
        OrchestratorClientContextConfiguration.class,
        ProdBlackboxContextConfiguration.class,
        TelemostTankerContextConfiguration.class,
        YandexStaffUserRegistryContextConfiguration.class,
        TelemostWorkerContextConfiguration.class,
})
public class TelemostServicesContextConfiguration {

    @Autowired
    private EnvironmentType environmentType;

    @Value("${telemost.xiva.mediator.service}")
    private String mediatorXivaService;

    @Value("${xiva.host}")
    private String xivaBaseUrl;

    @Value("${telemost.xiva.token}")
    private String token;

    @Value("${telemost.chat.avatar-ids}")
    private CommaSeparated chatAvatarIds;

    @Autowired
    private ZkPath zkRoot;

    private final Supplier<String> uuidWithoutDashValueProvider = UUIDUtils::generateUUIDHex;

    @Bean
    public UserTokenService userTokenService(UserTokenDtoDao userTokenDtoDao)
    {
        return new UserTokenService(userTokenDtoDao);
    }

    @Bean
    public YandexStaffUserRegistry yandexStaffUserRegistry(ZkManager zkManager)
    {
        YandexStaffUserRegistry registry = new YandexStaffUserRegistry(ZkUtils.rootPath("chemodan", environmentType).child("yandexoids"));
        zkManager.addClient(registry);
        return registry;
    }

    @Bean
    public ShardPropertiesZkRegistry shardPropertiesZkRegistry(
            @Value("${telemost.shard.id:-0}") int shardId,
            ZkManager zkManager
    ) {
        ShardPropertiesZkRegistry registry = new ShardPropertiesZkRegistry(zkRoot.child("telemost").child("shard-properties"), shardId);
        zkManager.addClient(registry);
        return registry;
    }

    @Bean
    public OverloadService overloadService(
            ShardPropertiesZkRegistry shardPropertiesZkRegistry,
            ConferenceUserDao conferenceUserDao,
            ConferenceService conferenceService,
            ConferenceUriService conferenceUriService
    ) {
        return new OverloadService(shardPropertiesZkRegistry, conferenceUserDao, conferenceService, conferenceUriService);
    }

    @Bean
    public StaffService staffService(YandexStaffUserRegistry yandexStaffUserRegistry)
    {
        return new StaffService(yandexStaffUserRegistry);
    }

    @Bean
    public ConferenceCreator defaultConferenceCreator(ConferenceUriService conferenceUriService)
    {
        return new DefaultConferenceCreator(uuidWithoutDashValueProvider, conferenceUriService);
    }

    @Bean
    public ConferenceCreator yaTeamConferenceCreator(ConferenceUriService conferenceUriService)
    {
        return new YaTeamConferenceCreator(uuidWithoutDashValueProvider, conferenceUriService);
    }

    @Bean
    public LimitsService limitsService()
    {
        return new LimitsService();
    }

    @Bean
    public PropertyManager propertyManager()
    {
        return new PropertyManagerImpl();
    }

    @Bean
    public ConferenceService conferenceKeyService(ConferenceDtoDao conferenceDtoDao,
                                                  ConferenceStateDao conferenceStateDao,
                                                  UserDao userDao,
                                                  ConferenceUserDao conferenceUserDao,
                                                  CalendarService calendarService,
                                                  TransactionTemplate transactionTemplate,
                                                  @Value("${conferences.creation.retry-count}") int insertConferenceRetryCount,
                                                  ConferenceUriService conferenceUriService,
                                                  UserTokenService userTokenService,
                                                  List<ConferenceCreator> conferenceCreators,
                                                  LimitsService limitsService, StaffService staffService,
                                                  BroadcastService broadcastService,
                                                  BroadcastUriService broadcastUriService,
                                                  ConferenceManagerClient conferenceManagerClient,
                                                  PropertyManager propertyManager,
                                                  @Value("${conferences.save.to.tcm}") boolean saveConferenceInfoToTcm)
    {
        return new ConferenceService(uuidWithoutDashValueProvider,
                conferenceDtoDao, conferenceStateDao, userDao, conferenceUserDao, calendarService, transactionTemplate,
                insertConferenceRetryCount, conferenceUriService, userTokenService, Cf.x(conferenceCreators),
                limitsService, staffService, broadcastService, broadcastUriService, conferenceManagerClient,
                propertyManager, saveConferenceInfoToTcm);
    }

    @Bean
    public ConferenceUriService conferenceUriService(@Value("${conference.uri.pattern.regexp}") String conferenceUriPattern,
                                                     @Value("${conference.uri.template}") String urlTemplate,
                                                     @Value("${conference.uri.ya-team.pattern.regexp}") String urlYateamPattern,
                                                     @Value("${conference.uri.host.default}") String uriHost,
                                                     @Value("${conference.uri.host.ya-team}") String yaTeamUriHost)
    {
        return new ConferenceUriService(
                () -> Random2.R.nextDigits(14),
                () -> Random2.R.nextDigits(38),
                conferenceUriPattern,
                urlTemplate,
                urlYateamPattern,
                uriHost,
                yaTeamUriHost);
    }

    @Bean
    public BroadcastUriService broadcastUriService(@Value("${broadcast.uri.pattern.regexp}") String broadcastUriPattern,
                                                   @Value("${broadcast.uri.template}") String urlTemplate,
                                                   @Value("${broadcast.uri.host.default}") String uriHost)
    {
        return new BroadcastUriService(broadcastUriPattern, urlTemplate, uriHost);
    }

    @Bean
    public WebsocketUriService websocketUriService()
    {
        return new WebsocketUriService(removeSchemeFromBaseUrl(xivaBaseUrl), mediatorXivaService);
    }

    @Bean
    @OverridableValuePrefix("xiva")
    public HttpClientConfigurator xivaHttpClientConfigurator()
    {
        return new HttpClientConfigurator();
    }

    @Bean
    public BasicXivaClient basicXivaClient(HttpClientConfigurator xivaHttpClientConfigurator)
    {
        return new BasicXivaClient(xivaBaseUrl, xivaHttpClientConfigurator.configure());
    }

    @Bean
    public RoomService roomServiceBlockingV2(BasicXivaClient basicXivaClient, ObjectMapper objectMapper,
                                             WebsocketUriService websocketUriService,
                                             @Qualifier("roomBlockingManager") RoomManager roomManager,
                                             ConferenceParticipantsService conferenceParticipantsService,
                                             BroadcastService broadcastService)
    {
        return new RoomService(mediatorXivaService, basicXivaClient, websocketUriService, objectMapper,
                roomManager, token, conferenceParticipantsService, ApiVersion.V2, broadcastService);
    }

    @Bean
    public RoomService roomServiceAsyncV1(BasicXivaClient basicXivaClient, ObjectMapper objectMapper,
                                          WebsocketUriService websocketUriService,
                                          @Qualifier("roomBlockingManager") RoomManager roomManager,
                                          ConferenceParticipantsService conferenceParticipantsService,
                                          BroadcastService broadcastService)
    {
        return new RoomService(mediatorXivaService, basicXivaClient, websocketUriService, objectMapper,
                roomManager, token, conferenceParticipantsService, ApiVersion.V1, broadcastService);
    }

    @Bean
    public ConferencePeerService conferencePeerService(Blackbox2 blackbox2,
                                                       @Value("${blackbox.avatars.url.prefix}") String avatarUrlPrefix)
    {
        return new ConferencePeerService(blackbox2, avatarUrlPrefix);
    }

    @Bean
    public ParticipantIdGenerator participantIdGenerator()
    {
        return new ParticipantIdGenerator();
    }

    @Bean
    public ConferenceParticipantsService conferenceParticipantsService(ConferencePeerService conferencePeerService,
                                                                       ConferencePeerDao conferencePeerDao,
                                                                       RoomGrpc.RoomBlockingStub roomBlockingStub,
                                                                       MediaSessionDao mediaSessionDao,
                                                                       LimitsService limitsService,
                                                                       ParticipantIdGenerator participantIdGenerator)
    {
        return new ConferenceParticipantsService(conferencePeerService, conferencePeerDao, roomBlockingStub, mediaSessionDao, limitsService,
                participantIdGenerator);
    }

    @Bean
    public ClientConfigurationService clientConfigurationService(ObjectMapper objectMapper,
                                                                 ClientConfigurationDao clientConfigurationDao,
                                                                 TransactionTemplate transactionTemplate)
    {
        return new ClientConfigurationService(objectMapper, clientConfigurationDao, transactionTemplate);
    }

    @Bean
    public HttpClientConfigurator chatHttpClientConfigurator()
    {
        return new HttpClientConfigurator();
    }

    @Bean
    public ChatParamsService getChatParamsService(@Value("${telemost.chat.type-id}") String chatTypeId,
                                                  @Value("${telemost.conference-chat.namespace-id}") String chatConferenceNamespaceId,
                                                  @Value("${telemost.conference-chat.is-public}") boolean chatConferenceIsPublic,
                                                  @Value("${telemost.broadcast-chat.namespace-id}") String chatBroadcastNamespaceId,
                                                  @Value("${telemost.broadcast-chat.is-public}") boolean chatBroadcastIsPublic)
    {
        return new ChatParamsService(chatTypeId, chatConferenceNamespaceId, chatConferenceIsPublic,
                chatBroadcastNamespaceId, chatBroadcastIsPublic);
    }

    @Bean
    public ChatClient getChatClient(@Value("${telemost.chat.base-meta-url}") URI chatBaseMetaUrl,
                                    @Value("${telemost.chat.base-fanout-url}") URI chatBaseFanoutUrl,
                                    @Value("${telemost.chat.namespace}") String chatNamespace,
                                    ChatParamsService chatParamsService)
    {
        return new ChatClientImpl(chatHttpClientConfigurator().configure(),
                chatBaseMetaUrl, chatBaseFanoutUrl, chatNamespace, chatParamsService);
    }

    @Bean
    public ChatService getChatService(ConferenceService conferenceService, ConferenceUriService conferenceUriService,
                                      ChatClient chatClient, ConferenceDtoDao conferenceDtoDao,
                                      ConferenceStateDao conferenceStateDao,
                                      AppMessageSender appMessageSender, ConferencePeerService conferencePeerService,
                                      CalendarService calendarService, BroadcastService broadcastService,
                                      BroadcastUriService broadcastUriService, StreamService streamService,
                                      @Value("${telemost.chat.bot-guid}") String bot, TankerClient tankerClient,
                                      ConferenceUserDao conferenceUserDao)
    {
        return new ChatService(conferenceService, conferenceUriService, chatClient, conferenceDtoDao, conferenceUserDao,
                conferenceStateDao, appMessageSender, conferencePeerService, calendarService, broadcastService,
                broadcastUriService, streamService, chatAvatarIds.getList(), UUID.fromString(bot), tankerClient);
    }

    @Bean
    public CommandService commandService(ConferencePeerDao conferencePeerDao, AppMessageSender appMessageSender,
                                         ConferenceUserDao conferenceUserDao, UserStateDtoDao userStateDtoDao,
                                         ConferenceService conferenceService, ChatService chatService)
    {
        return new CommandService(
                conferencePeerDao, userStateDtoDao, appMessageSender,
                conferenceUserDao, conferenceService, chatService);
    }

    @Bean
    public BroadcastService broadcastService(@Value("${broadcasts.creation.retry-count}") int insertBroadcastRetryCount,
                                               BroadcastUriService broadcastUriService,
                                               BroadcastDao broadcastDao, StreamDao streamDao)
    {
        return new BroadcastService(insertBroadcastRetryCount, broadcastUriService, broadcastDao, streamDao);
    }

    @Bean
    public StreamService streamService(@Value("${broadcasts.creation.retry-count}") int insertBroadcastRetryCount,
                                       BroadcastService broadcastService, StreamDao streamDao,
                                       BroadcastUriService broadcastUriService, ConferenceService conferenceService,
                                       TranslatorsOrchestrator translatorsOrchestrator, TranslatorClient translatorClient,
                                       UgcLiveClient ugcLiveClient, UgcLiveService ugcLiveService,
                                       CalendarService calendarService, ConferenceStateDao conferenceStateDao,
                                       AppMessageSender appMessageSender, ConferencePeerService conferencePeerService)
    {
        return new StreamService(
                insertBroadcastRetryCount, broadcastService, streamDao, conferenceService,
                broadcastUriService, translatorsOrchestrator, translatorClient,
                ugcLiveClient, ugcLiveService, calendarService, conferenceStateDao, appMessageSender,
                conferencePeerService);
    }

    @Bean
    public TranslatorsOrchestrator translatorsOrchestrator(
            OrchestratorClient orchestratorClient)
    {
        return new TranslatorsOrchestratorImpl(orchestratorClient);
    }

    @Bean
    @OverridableValuePrefix("ugc-live")
    public HttpClientConfigurator ugcLiveHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    public UserTicketSupplier ugcLiveUserTicketSupplier(@Value("${ugc-live.user.token}") String token,
                                                        @Value("${ugc-live.user.env}") BlackboxType type, Tvm2 tvm2)
    {
        if (!Cf.list(EnvironmentType.DEVELOPMENT, EnvironmentType.TESTS).containsTs(environmentType)) {
            Check.notEmpty(token, "Missing ugc-live.user.token");
        }
        return new OAuthUserTicketSupplier(token, type, tvm2).caching();
    }

    @Bean
    public UgcLiveClient ugcLiveClient(
            @Value("${ugc-live.base-url}") URI baseUrl,
            @Value("${ugc-live.channel-id}") String channelId,
            UserTicketSupplier ugcLiveUserTicketSupplier)
    {
        return new UgcLiveClientImpl(
                ugcLiveHttpClientConfigurator().configure(),
                baseUrl, channelId, ugcLiveUserTicketSupplier);
    }

    @Bean
    public UgcLiveStreamPublisher ugcLiveStreamPublisher(UgcLiveClient ugcLiveClient)
    {
        return new UgcLiveStreamPublisherImpl(ugcLiveClient);
    }

    @Bean
    public UgcLiveService ugcLiveService(UgcLiveClient client, UgcLiveStreamPublisher publisher) {
        return new UgcLiveService(client, publisher);
    }

    @Bean
    @OverridableValuePrefix("translator")
    public HttpClientConfigurator translatorHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    public TranslatorClient translatorClient()
    {
        return new TranslatorClientImpl(translatorHttpClientConfigurator().configure());
    }

    @Bean
    public UserService userService(@Value("${users.insert.retry-count}") int insertUserRetryCount,
                                   UserDao userDao, StaffService staffService)
    {
        return new UserService(insertUserRetryCount, userDao, staffService);
    }

    @Bean
    public HttpClientConfigurator calendarHttpClientConfigurator()
    {
        return new HttpClientConfigurator();
    }

    @Bean
    public CalendarClient getCalendarClient(@Value("${telemost.calendar.base-corp-url}") URI calendarBaseCorpUrl,
                                            @Value("${telemost.calendar.base-public-url}") URI calendarBasePublicUrl)
    {
        return new CalendarClientImpl(calendarHttpClientConfigurator().configure(),
                calendarBaseCorpUrl, calendarBasePublicUrl);
    }

    @Bean
    public CalendarService getCalendarService(CalendarClient calendarClient, ConferenceUserDao conferenceUserDao)
    {
        return new CalendarService(calendarClient, conferenceUserDao);
    }

    @Bean
    public PeerStateService getPeerStateService(ObjectMapper objectMapper, ConferencePeerService conferencePeerService,
                                                UserStateDtoDao userStateDtoDao, ConferencePeerDao conferencePeerDao)
    {
        return new PeerStateService(conferencePeerService, objectMapper, userStateDtoDao, conferencePeerDao);
    }

    @Bean
    @OverridableValuePrefix("tcm")
    public HttpClientConfigurator tcmClientConfigurator()
    {
        return new HttpClientConfigurator();
    }

    @Bean
    public ConferenceManagerClient conferenceManagerClient(HttpClientConfigurator tcmClientConfigurator,
                                                           @Value("${tcm.base-url}") URI tcmBaseUrl,
                                                           @Value("${telemost.shard.id:-0}") int shardId,
                                                           @Value("${tcm.safe.call:-false}") boolean tcmSafeCall,
                                                           ObjectMapper objectMapper)
    {
        return new ConferenceManagerClient(tcmClientConfigurator.configure(), tcmBaseUrl, shardId, objectMapper, tcmSafeCall);
    }

    private String removeSchemeFromBaseUrl(String baseUrl)
    {
        return baseUrl.replaceAll("^http(s*)://", "");
    }
}
