package ru.yandex.chemodan.app.telemost.config.common;

import java.util.Collection;
import java.util.Optional;
import java.util.Properties;

import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.boot.CoolPingServletContextConfiguration;
import ru.yandex.chemodan.boot.admin.AdminBenderConfigurator;
import ru.yandex.chemodan.boot.admin.AdminUtil;
import ru.yandex.chemodan.boot.admin.CustomAdminActionInterceptorsHolder;
import ru.yandex.chemodan.boot.admin.DbsMetrics;
import ru.yandex.chemodan.boot.admin.LogRequestServlet;
import ru.yandex.chemodan.http.DiskJetty;
import ru.yandex.chemodan.log.DiskLog4jRequestLog;
import ru.yandex.chemodan.util.DiskHostnameUtils;
import ru.yandex.chemodan.util.ping.CoolPingServlet;
import ru.yandex.commune.a3.action.intercept.ActionInvocationInterceptor;
import ru.yandex.commune.admin.web.Admin;
import ru.yandex.commune.admin.web.AdminApp;
import ru.yandex.commune.admin.web.AdminBender;
import ru.yandex.commune.admin.web.AdminConfigurator;
import ru.yandex.commune.admin.web.AdminGroups;
import ru.yandex.commune.admin.web.ToyServletsContextConfiguration;
import ru.yandex.commune.admin.web.auth.AdminAuthenticationConfiguration;
import ru.yandex.commune.admin.web.bender.DateTimeMarshaller;
import ru.yandex.commune.admin.web.bender.DurationMarshaller;
import ru.yandex.commune.admin.web.bender.InstantMarshaller;
import ru.yandex.commune.admin.web.bureaucracy.ResourcesAdminPage;
import ru.yandex.commune.admin.web.envVar.EnvVarAdminPage;
import ru.yandex.commune.admin.web.info.InfoAdminPage;
import ru.yandex.commune.admin.web.info.InfoServlet;
import ru.yandex.commune.admin.web.manager.ManagerContextConfiguration;
import ru.yandex.commune.admin.web.properties.PropertiesAdminPage;
import ru.yandex.commune.admin.web.properties.PropertiesServlet;
import ru.yandex.commune.admin.web.script.ScriptContextConfiguration;
import ru.yandex.commune.admin.web.settings.SettingsAdminPage;
import ru.yandex.commune.admin.web.support.AdminMultilineTextTemplateEngine;
import ru.yandex.commune.admin.web.thread.ThreadAdminPage;
import ru.yandex.commune.admin.web.version.VersionHolder;
import ru.yandex.commune.admin.web.version.VersionServlet;
import ru.yandex.commune.admin.web.workerBeans.WorkerBeansAdminPage;
import ru.yandex.commune.admin.z.ZControllerInitializer;
import ru.yandex.commune.alive2.AliveAppInfo;
import ru.yandex.commune.alive2.admin.AliveAppsAdminConfiguration;
import ru.yandex.commune.db.admin.dbs.DbsAdminPage;
import ru.yandex.commune.db.admin.dbs.DbsContextConfiguration;
import ru.yandex.commune.db.admin.sql.SqlContextConfiguration;
import ru.yandex.commune.monitoring.SensorServlet;
import ru.yandex.commune.util.serialize.ToMultilineSerializer;
import ru.yandex.commune.util.serialize.ToMultilineSerializerContextConfiguration;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bureaucracy.resources.ExternalResourceDescription;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.email.bender.EmailMarshaller;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.jvm.ThreadsServlet;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.spring.ApplicationContextUtils;
import ru.yandex.misc.version.AppName;
import ru.yandex.misc.version.Version;
import ru.yandex.misc.web.servlet.PidServlet;
import ru.yandex.misc.web.servlet.WtdFilter;
import ru.yandex.misc.web.servletContainer.SingleWarJetty;


@Configuration
@Import({
        CoolPingServletContextConfiguration.class,
        TelemostIdmInterceptorContextConfiguration.class,
        ManagerContextConfiguration.class,
        ScriptContextConfiguration.class,
        DbsContextConfiguration.class,
        SqlContextConfiguration.class,
        ToyServletsContextConfiguration.class,
        ToMultilineSerializerContextConfiguration.class
})
public class TelemostAdminDaemonContextConfiguration {

    @Autowired
    private ApplicationContext applicationContext;

    @Bean
    @Admin
    public SingleWarJetty adminJetty(
            @Value("${admin.http.port}") int adminHttpPort,
            @Value("${http.maxThreads}") int maxThreads,
            @Value("${http.maxQueueLength}") int maxQueueLength,
            @Value("${debug.log.servlet.enabled:-false}") boolean debugLogServletEnabled,
            CoolPingServlet coolPingServlet) {
        SingleWarJetty jetty = new DiskJetty("telemost-admin");
        jetty.setHttpPort(adminHttpPort);
        jetty.setMaxThreads(maxThreads);
        jetty.setMaxQueueLength(maxQueueLength);
        jetty.setCopyBeanPostprocessors(true);
        jetty.setLookupServletsInContext(false);
        jetty.setRequestLogFactory(DiskLog4jRequestLog.consF());
        jetty.addFilterMapping("/*", new WtdFilter());
        if (debugLogServletEnabled) {
            jetty.addServletMapping("/log", new LogRequestServlet());
        }
        jetty.addServletMapping("/pid", new PidServlet());
        jetty.addServletMapping("/ping", coolPingServlet);
        return jetty;
    }

    @Admin
    @Bean
    public AdminApp adminApp(
            Optional<Collection<CustomAdminActionInterceptorsHolder>> customAdminActionInterceptorsHolders,
            Optional<AdminAuthenticationConfiguration> adminAuthenticationConfiguration,
            @Admin SingleWarJetty adminJetty,
            ToMultilineSerializer toMultilineSerializer,
            Optional<AdminBenderConfigurator> adminBenderConfigurator,
            AliveAppInfo aliveAppInfo) {
        CustomMarshallerUnmarshallerFactoryBuilder builder = CustomMarshallerUnmarshallerFactoryBuilder.cons()
                .add(DateTime.class, new DateTimeMarshaller())
                .add(Instant.class, new InstantMarshaller())
                .add(Duration.class, new DurationMarshaller())
                .add(Email.class, new EmailMarshaller());

        adminBenderConfigurator.ifPresent(c -> c.configure(builder));

        BenderMapper mapper = new BenderMapper(new BenderConfiguration(AdminBender.SETTINGS, builder.build()));

        ListF<ActionInvocationInterceptor> customInterceptors = customAdminActionInterceptorsHolders
                .orElseGet(Cf::list)
                .stream()
                .map(CustomAdminActionInterceptorsHolder::getInterceptors)
                .reduce(ListF::plus)
                .orElseGet(Cf::list);

        String title = Option.ofNullable(System.getenv("QLOUD_DISCOVERY_INSTANCE")).filterNot(String::isEmpty)
                .getOrElse(aliveAppInfo.getAppName() + "[" + EnvironmentType.getActive().getValue() + "]");

        return AdminConfigurator.configure(
                "/z",
                adminJetty::addServletMapping,
                adminJetty::addFilterMapping,
                mapper,
                Cf.list(new AdminMultilineTextTemplateEngine(mapper, toMultilineSerializer)),
                Option.x(adminAuthenticationConfiguration),
                customInterceptors,
                Option.of(EnvironmentType.getActive()),
                Option.of(title),
                AdminUtil.sshLink(aliveAppInfo));
    }

    // We need to populate this bean because it's Service: it compiles all xsl templates.
    @Bean
    public ZControllerInitializer controllerValidator(AdminApp adminApp) {
        return adminApp.createControllerInitializer();
    }

    @Bean
    public SensorServlet sensorServlet(@Admin SingleWarJetty adminJetty) {
        SensorServlet servlet = new SensorServlet();
        adminJetty.addServletMapping("/sensor/*", servlet);
        return servlet;
    }

    @Bean
    public VersionHolder versionHolder(Optional<Version> version) {
        return new VersionHolder(Option.x(version));
    }

    @Bean
    public AliveAppsAdminConfiguration aliveAppsAdminConfiguration(
            @Value("${admin.yateam.proxy.host}") String yaTeamProxyHost
    ) {
        return new AliveAppsAdminConfiguration(
                AliveAppsAdminConfiguration.GroupBy.APP,
                app -> DiskHostnameUtils.isDisk(app.getHostname()),
                app -> false,
                hostname -> {
                    if (!yaTeamProxyHost.isEmpty()) {
                        if (!hostname.endsWith("qloud-c.yandex.net")) {
                            return hostname;
                        }
                        return StringUtils.substringBefore(hostname, "qloud-c.yandex.net") + yaTeamProxyHost;
                    } else {
                        return hostname;
                    }
                },
                hostname -> {
                    if (!yaTeamProxyHost.isEmpty() && hostname.endsWith("qloud-c.yandex.net")) {
                        return 80;
                    } else {
                        return 81;
                    }
                });
    }

    @Bean
    public Void configure(
            ApplicationContext applicationContext,
            AdminApp adminApp,
            AppName appName,
            @Admin SingleWarJetty adminJetty,
            Properties properties,
            VersionHolder versionHolder) {
        Instant startup = new Instant(applicationContext.getStartupDate());
        adminApp.registerAdminPage(
                AdminGroups.INFO, new InfoAdminPage(appName, versionHolder, startup));
        adminApp.registerPlain("/info", new InfoServlet(versionHolder, startup));

        adminApp.registerHiddenAdminPage(new SettingsAdminPage());

        adminJetty.addServletMapping("/version/*", new VersionServlet(versionHolder));

        Function0<ListF<ExternalResourceDescription>> getF = ApplicationContextUtils
                .<ExternalResourceDescription>beansOfTypeF()
                .bind1(applicationContext)
                .bind(ExternalResourceDescription.class);
        adminApp.registerAdminPage(AdminGroups.INFO, new ResourcesAdminPage(getF));

        adminApp.registerAdminPage(AdminGroups.INFO, new PropertiesAdminPage(properties));
        adminApp.registerPlain("/properties", new PropertiesServlet(properties));

        adminApp.registerAdminPage(AdminGroups.INFO, new EnvVarAdminPage());

        adminApp.registerAdminPage(AdminGroups.INFO, new ThreadAdminPage());
        adminApp.registerPlain("/threads", new ThreadsServlet());

        adminApp.registerAdminPage(new WorkerBeansAdminPage(applicationContext));

        return null;
    }

    @Bean
    public DbsMetrics dbsMetrics() {
        return new DbsMetrics(new DbsAdminPage(applicationContext, Option.empty()));
    }
}
