package ru.yandex.chemodan.app.telemost.logging;

import java.io.Serializable;

import org.apache.logging.log4j.core.Layout;
import org.apache.logging.log4j.core.layout.PatternLayout;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.RequestLog;
import org.eclipse.jetty.server.Response;
import org.eclipse.jetty.util.component.AbstractLifeCycle;

import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.http.RequestNdcUtil;
import ru.yandex.chemodan.log.Log4jHelper;
import ru.yandex.chemodan.log.utils.TskvEscapeUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.misc.time.TimeUtils;
import ru.yandex.misc.version.AppName;
import ru.yandex.misc.web.servlet.HttpRequestUtils;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

public class TelemostAccessLog extends AbstractLifeCycle implements RequestLog {

    private static final Logger logger = LoggerFactory.getLogger(Log4jHelper.accessLogTskvName);

    public static Layout<? extends Serializable> accessLayout(AppName applicationName) {
        String pattern = "tskv"
                + "\ttskv_format=ydisk-" + applicationName.appName() + "-log"
                + "\thost=" + HostnameUtils.localHostname()
                + "\tname=" + applicationName.appName() + ".access"
                + "\tappname=" + applicationName.appName()
                + "\tunixtime=%d{UNIX}"
                + "\ttimestamp=%d{yyyy-MM-dd HH:mm:ss,SSS}"
                + "\ttimezone=%d{XX}"
                + "\t%m%n";
        return PatternLayout.newBuilder().withPattern(pattern).build();
    }

    @Override
    public void log(Request request, Response response) {
        HttpServletRequestX reqx = HttpServletRequestX.wrap(request);
        String requestTime = TimeUtils.millisecondsToSecondsStringToNow(request.getTimeStamp());

        Option<String> uid = Option.ofNullable(request.getAttribute("request-uid"))
                .map(Object::toString);

        Option<MapF<String, String>> ndc = RequestNdcUtil.getNdcO(request);
        Option<String> rid = ndc.flatMapO(x -> x.getO("rid"));
        Option<String> ycrid = ndc.flatMapO(x -> x.getO("ycrid"));

        String headers = reqx.getHeaders().map(HttpRequestUtils::encodeSensitiveHeader)
                .map(x -> String.format("\"%s: %s\"", x._1, x._2)).mkString(" ");

        Tuple2List<String, String> fields = Tuple2List.fromPairs(
                "ycrid", ycrid.getOrElse("-"),
                "request_id", rid.getOrElse("-"),
                "pid", Thread.currentThread().getName(),
                "module", "-",
                "headers", headers,
                "method", request.getMethod(),
                "uid", uid.getOrElse("-"),
                "uri", request.getRequestURI() + (request.getQueryString() == null ? "" : "?" + request.getQueryString()),
                "proto", request.getProtocol(),
                "status", String.valueOf(response.getStatus()),
                "request_time", requestTime);

        logger.info(fields.map((key, value) -> key + "=" +
                TskvEscapeUtils.escape(value, TskvEscapeUtils.DEFAULT_SYMBOL_ESCAPES_WITHOUT_QUOTES)).mkString("\t"));
    }
}
