package ru.yandex.chemodan.app.telemost.repository.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;
import java.util.stream.Collectors;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.telemost.exceptions.TelemostRuntimeException;
import ru.yandex.chemodan.app.telemost.repository.dao.BroadcastDao;
import ru.yandex.chemodan.app.telemost.repository.model.BroadcastDto;
import ru.yandex.chemodan.app.telemost.services.BroadcastUriService;
import ru.yandex.chemodan.app.telemost.services.ChatParamsService;
import ru.yandex.chemodan.app.telemost.services.model.ChatType;
import ru.yandex.chemodan.app.telemost.web.v2.model.BroadcastStatus;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class BroadcastPgDaoImpl extends AbstractPgUUIDKeyDao<BroadcastDto> implements BroadcastDao {

    private static final String TABLE_NAME = "telemost.broadcasts";

    private static final String ID_FIELD = "id";
    private static final String CONFERENCE_ID_FIELD = "conference_id";
    private static final String CREATED_BY_FIELD = "created_by";
    private static final String BROADCAST_KEY_FIELD = "broadcast_key";
    private static final String BROADCAST_CHAT_ID_FIELD = "broadcast_chat_id";
    private static final String CAPTION_FIELD = "caption";
    private static final String DESCRIPTION_FIELD = "description";
    private static final String UGC_LIVE_LINE_ID_FIELD = "ugc_live_line_id";
    private static final String STATUS_FIELD = "status";

    private static final ListF<String> FIELDS_TO_INSERT = Cf.list(
            CONFERENCE_ID_FIELD, CREATED_BY_FIELD, BROADCAST_KEY_FIELD, BROADCAST_CHAT_ID_FIELD,
            CAPTION_FIELD, DESCRIPTION_FIELD, UGC_LIVE_LINE_ID_FIELD);
    private static final ListF<String> FIELDS_TO_SELECT = Cf.list(ID_FIELD).plus(FIELDS_TO_INSERT);

    private final BroadcastUriService broadcastUriService;
    private final ChatParamsService chatParamsService;

    public BroadcastPgDaoImpl(JdbcTemplate3 jdbcTemplate, BroadcastUriService broadcastUriService,
                                ChatParamsService chatParamsService) {
        super(jdbcTemplate);

        this.broadcastUriService = broadcastUriService;
        this.chatParamsService = chatParamsService;
    }

    @Override
    public void insertIfNotExists(BroadcastDto broadcast) {
        getJdbcTemplate().update(Queries.INSERT_IF_NOT_EXISTS.query,
                Cf.toMap(Tuple2List.fromPairs(
                        CONFERENCE_ID_FIELD, broadcast.getConferenceId(),
                        CREATED_BY_FIELD, broadcast.getCreatedBy(),
                        BROADCAST_KEY_FIELD, broadcast.getBroadcastKey(),
                        BROADCAST_CHAT_ID_FIELD, broadcast.getBroadcastChatId(),
                        CAPTION_FIELD, broadcast.getCaption().orElse((String) null),
                        DESCRIPTION_FIELD, broadcast.getDescription().orElse((String) null),
                        UGC_LIVE_LINE_ID_FIELD, broadcast.getUgcLiveLineId().orElse((Long) null))));
    }

    @Override
    public BroadcastDto upsert(BroadcastDto broadcast) {
        return getJdbcTemplate().query(Queries.UPSERT.query, (rs, rowNum) -> parseRow(rs),
                Cf.toMap(Tuple2List.fromPairs(
                        CONFERENCE_ID_FIELD, broadcast.getConferenceId(),
                        CREATED_BY_FIELD, broadcast.getCreatedBy(),
                        BROADCAST_KEY_FIELD, broadcast.getBroadcastKey(),
                        BROADCAST_CHAT_ID_FIELD, broadcast.getBroadcastChatId().orElse((UUID) null),
                        CAPTION_FIELD, broadcast.getCaption().orElse((String) null),
                        DESCRIPTION_FIELD, broadcast.getDescription().orElse((String) null),
                        UGC_LIVE_LINE_ID_FIELD, broadcast.getUgcLiveLineId().orElse((Long) null)))).get(0);
    }

    @Override
    public boolean updateUgcLiveLineIfNull(UUID broadcastId, long lineId) {
        return getJdbcTemplate().update(Queries.UPDATE_LINE_IF_NULL.query,
                Cf.map(ID_FIELD, broadcastId, UGC_LIVE_LINE_ID_FIELD, lineId)
        ) > 0;
    }

    @Override
    public boolean updateBroadcastChatIdIfNull(UUID broadcastId, UUID broadcastChatId) {
        return getJdbcTemplate().update(Queries.UPDATE_CHAT_ID_IF_NULL.query,
                Cf.map(ID_FIELD, broadcastId, BROADCAST_CHAT_ID_FIELD, broadcastChatId)
        ) > 0;
    }

    @Override
    public Option<BroadcastDto> findByBroadcastId(UUID broadcastId) {
        return getJdbcTemplate().query(Queries.SELECT_BY_ID.query, (rs, rowNum) -> parseRow(rs),
                Cf.map(ID_FIELD, broadcastId)).firstO();
    }

    @Override
    public Option<BroadcastDto> findByBroadcastKey(String broadcastKey) {
        return getJdbcTemplate().query(Queries.SELECT_BY_KEY.query, (rs, rowNum) -> parseRow(rs),
                Cf.map(BROADCAST_KEY_FIELD, broadcastKey)).firstO();
    }

    @Override
    public Option<BroadcastDto> findByConferenceId(UUID conferenceId) {
        return getJdbcTemplate().query(Queries.SELECT_BY_CONFERENCE_ID.query, (rs, rowNum) -> parseRow(rs),
                Cf.map(CONFERENCE_ID_FIELD, conferenceId)).firstO();
    }

    @Override
    protected String getTableName() {
        return TABLE_NAME;
    }

    @Override
    protected ListF<String> getFieldsToSelect() {
        return FIELDS_TO_SELECT;
    }

    @Override
    protected BroadcastDto parseRow(ResultSet rs) {
        try {
            String broadcastKey = rs.getString(BROADCAST_KEY_FIELD);
            Option<UUID> chatIdO = Option.ofNullable(rs.getString(BROADCAST_CHAT_ID_FIELD)).map(UUID::fromString);
            return new BroadcastDto(
                    Option.of(UUID.fromString(rs.getString(ID_FIELD))),
                    UUID.fromString(rs.getString(CONFERENCE_ID_FIELD)),
                    rs.getString(CREATED_BY_FIELD),
                    broadcastKey,
                    chatIdO,
                    Option.ofNullable(rs.getString(CAPTION_FIELD)),
                    Option.ofNullable(rs.getString(DESCRIPTION_FIELD)),
                    Option.ofNullable(rs.getObject(UGC_LIVE_LINE_ID_FIELD, Long.class)),
                    broadcastUriService.buildBroadcastUrl(broadcastKey),
                    chatParamsService.chatPath(chatIdO, ChatType.BROADCAST),
                    BroadcastStatus.valueOf(rs.getString(STATUS_FIELD))
            );
        } catch (SQLException e) {
            throw new TelemostRuntimeException(e);
        }
    }

    private enum Queries {
        SELECT("SELECT " + FIELDS_TO_SELECT.map(field -> "t." + field).mkString(",") + "," +
                " CASE WHEN s.id IS NULL THEN 'CREATED'" +
                "      WHEN s.stopped_at IS NULL THEN 'STARTED'" +
                "      ELSE 'FINISHED'" +
                " END AS status" +
                " FROM " + TABLE_NAME + " AS t LEFT JOIN telemost.streams AS s" +
                " ON t." + ID_FIELD + " = s.broadcast_id"),
        SELECT_BY_ID(SELECT.query +
                " WHERE t." + ID_FIELD + " = :id" +
                " ORDER BY s.started_at DESC LIMIT 1"),
        SELECT_BY_KEY(SELECT.query +
                " WHERE t." + BROADCAST_KEY_FIELD + " = :broadcast_key" +
                " ORDER BY s.started_at DESC LIMIT 1"),
        SELECT_BY_CONFERENCE_ID(SELECT.query +
                " WHERE t." + CONFERENCE_ID_FIELD + " = :conference_id" +
                " ORDER BY s.started_at DESC LIMIT 1"),
        INSERT("INSERT INTO " + TABLE_NAME + " (" + FIELDS_TO_INSERT.mkString(",") + ")" +
                " VALUES (" + FIELDS_TO_INSERT.stream().map(str -> ":" + str).collect(Collectors.joining(",")) + ")"),
        UPSERT("WITH t AS (" +
                INSERT.query +
                " ON CONFLICT(conference_id) DO UPDATE SET broadcast_key = :broadcast_key, broadcast_chat_id = :broadcast_chat_id" +
                " , caption = :caption, description = :description" +
                " RETURNING *) " + SELECT_BY_KEY.query.replace(TABLE_NAME + " AS ", "")),
        UPDATE_LINE_IF_NULL("UPDATE " + TABLE_NAME +
                " SET " + UGC_LIVE_LINE_ID_FIELD + " = :" + UGC_LIVE_LINE_ID_FIELD +
                " WHERE " + ID_FIELD + " = :" + ID_FIELD + " AND " + UGC_LIVE_LINE_ID_FIELD + " IS NULL"),
        UPDATE_CHAT_ID_IF_NULL("UPDATE " + TABLE_NAME  +
                " SET " + BROADCAST_CHAT_ID_FIELD + " = :" + BROADCAST_CHAT_ID_FIELD +
                " WHERE " + ID_FIELD + " = :" + ID_FIELD + " AND " + BROADCAST_CHAT_ID_FIELD + " IS NULL"),
        INSERT_IF_NOT_EXISTS(INSERT.query +
                " ON CONFLICT(conference_id) DO NOTHING");

        private final String query;

        Queries(String query) {
            this.query = query;
        }
    }

}
