package ru.yandex.chemodan.app.telemost.repository.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.telemost.exceptions.TelemostRuntimeException;
import ru.yandex.chemodan.app.telemost.repository.dao.ClientConfigurationDao;
import ru.yandex.chemodan.app.telemost.repository.model.ClientConfigurationDto;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class ClientConfigurationPgDaoImpl extends AbstractPgUUIDKeyDao<ClientConfigurationDto>
        implements ClientConfigurationDao
{

    private static final ListF<String> FIELDS_TO_SELECT =
            Cf.list("id", "created_at", "created_by", "comment", "is_actual", "value");

    private static final String GET_ACTUAL = "get-actual";

    private static final String INSERT_ACTUAL = "insert-actual";

    private static final String EXPIRE_CURRENT = "expire-current";

    private static final Function<String, String> jsonFieldMapper = field -> !"value".equals(field) ? field : "value::text";

    private final MapF<String, String> queries = Cf.map(
            GET_ACTUAL, "SELECT " + getFieldsToSelect().map(jsonFieldMapper).mkString(", ") +
                    " FROM " + getTableName() +
                    " WHERE is_actual",
            INSERT_ACTUAL, "INSERT INTO " + getTableName() + " (created_at, created_by, comment, is_actual, value) " +
                    "VALUES (:created_at, :created_by, :comment, :is_actual, :value::jsonb) " +
                    "RETURNING " + getFieldsToSelect().map(jsonFieldMapper).mkString(", "),
            EXPIRE_CURRENT, "UPDATE " + getTableName() + " SET is_actual = false WHERE is_actual " +
                    "RETURNING " + getFieldsToSelect().map(jsonFieldMapper).mkString(", ")
    );

    public ClientConfigurationPgDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    protected String getTableName() {
        return "telemost.client_configuration";
    }

    @Override
    protected ListF<String> getFieldsToSelect() {
        return FIELDS_TO_SELECT;
    }

    @Override
    protected ClientConfigurationDto parseRow(ResultSet resultSet) {
        try {
            return new ClientConfigurationDto(Option.of(UUID.fromString(resultSet.getString("id"))),
                    new Instant(resultSet.getTimestamp("created_at")),
                    resultSet.getString("created_by"),
                    Option.ofNullable(resultSet.getString("comment")),
                    resultSet.getBoolean("is_actual"),
                    resultSet.getString("value"));
        } catch (SQLException e) {
            throw new TelemostRuntimeException(e);
        }
    }

    @Override
    public Option<ClientConfigurationDto> getActualConfiguration() {
        return getJdbcTemplate()
                .queryForOption(queries.getTs(GET_ACTUAL), (rs, rowNum) -> parseRow(rs));
    }

    @Override
    public ClientConfigurationDto insertActualConfiguration(ClientConfigurationDto clientConfigurationDto) {
        MapF<String, Object> params = Cf.map(
                "created_at", clientConfigurationDto.getCreatedAt(),
                "created_by", clientConfigurationDto.getCreatedBy(),
                "comment", clientConfigurationDto.getComment().getOrElse(() -> null),
                "is_actual", true
        );
        params = params
                .plus1("value", clientConfigurationDto.getValue());
        return getJdbcTemplate().query(queries.getTs(INSERT_ACTUAL), (rs, rowNum) -> parseRow(rs), params).first();
    }

    @Override
    public Option<ClientConfigurationDto> expireActualConfiguration() {
        return getJdbcTemplate().queryForOption(queries.getTs(EXPIRE_CURRENT), (rs, rowNum) -> parseRow(rs));
    }
}
