package ru.yandex.chemodan.app.telemost.repository.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.telemost.exceptions.TelemostRuntimeException;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceDtoDao;
import ru.yandex.chemodan.app.telemost.repository.model.ApiVersion;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceDto;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class ConferenceDtoPgDaoImpl extends AbstractPgUUIDKeyDao<ConferenceDto> implements ConferenceDtoDao {

    private static final ListF<String> FIELDS_TO_SELECT =
            Cf.list("id", "conference_id", "conference_password", "limit_type",
                    "created_at", "short_url_id", "owner_uid", "staff_only", "is_permanent", "is_yateam",
                    "event_id");

    private static final String INSERT_ONE = "insert-one";
    private static final String FIND_BY_SHORT_URL_ID = "find-by-short-url-id";
    private static final String FIND_BY_BROADCAST_KEY = "find-by-broadcast-key";
    private static final String FIND_BY_CONF_ID = "find-by-conf-id";
    private static final String LINK_TO_CALENDAR_EVENT = "link-to-calendar-event";
    private static final String GET_PEERS_VERSIONS = "peers_versions";

    private final MapF<String, String> queries = Cf
            .map(INSERT_ONE, "INSERT INTO " + getTableName() +
                    " (conference_id, conference_password, limit_type, created_at, short_url_id, staff_only, is_permanent, is_yateam, event_id) " +
                    "VALUES (:conference_id, :conference_password, :limit_type, :created_at, :short_url_id, :staff_only, :is_permanent, :is_yateam, :event_id) " +
                    "RETURNING id")
            .plus1(FIND_BY_SHORT_URL_ID, "SELECT " + FIELDS_TO_SELECT.mkString(", ") +
                    " FROM " + getTableName() +
                    " WHERE short_url_id = :short_url_id")
            .plus1(FIND_BY_BROADCAST_KEY, "SELECT " + FIELDS_TO_SELECT.map(field -> "c." + field).mkString(",") +
                    " FROM " + getTableName() + " c" +
                    " JOIN telemost.broadcasts b" +
                    " ON c.id = b.conference_id" +
                    " WHERE b.broadcast_key = :broadcast_key")
            .plus1(LINK_TO_CALENDAR_EVENT,
                    "UPDATE " + getTableName() + " SET event_id = :event_id WHERE id = :id")
            .plus1(FIND_BY_CONF_ID, "SELECT " + FIELDS_TO_SELECT.mkString(", ") +
                    " FROM " + getTableName() +
                    " WHERE conference_id = :conference_id")
            .plus1(GET_PEERS_VERSIONS, "select distinct users.api_version from " + getTableName() + " as conf " +
                    " join telemost.user_data users on users.conference_id = conf.id " +
                    " where conf.conference_id = :conf_id and users.api_version is not null");

    public ConferenceDtoPgDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public ConferenceDto insert(ConferenceDto conference) {
        MapF<String, Object> params = Cf.map(
                "conference_id", conference.getConferenceId(),
                "conference_password", conference.getConferencePassword(),
                "limit_type", conference.getLimitType()
        );
        params = params
                .plus1("short_url_id", conference.getShortUrlId())
                .plus1("staff_only", conference.isStaffOnly())
                .plus1("created_at", conference.getCreatedAt())
                .plus1("is_permanent", conference.isPermanent())
                .plus1("is_yateam", conference.isYaTeam())
                .plus1("event_id", conference.getEventId().getOrElse(() -> null));
        UUID id = getJdbcTemplate().query(queries.getTs(INSERT_ONE), newIdMapper(), params).first();
        return conference.withId(id);
    }

    @Override
    public Option<ConferenceDto> findByConferenceId(String conferenceId) {
        MapF<String, ?> params = Cf.map("conference_id", conferenceId);
        return getJdbcTemplate()
                .queryForOption(queries.getTs(FIND_BY_CONF_ID), (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public Option<ConferenceDto> findByShortUriIdO(String shortUriId) {
        MapF<String, ?> params = Cf.map("short_url_id", shortUriId);
        return getJdbcTemplate()
                .queryForOption(queries.getTs(FIND_BY_SHORT_URL_ID), (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public Option<ConferenceDto> findByBroadcastKeyO(String broadcastKey) {
        MapF<String, ?> params = Cf.map("broadcast_key", broadcastKey);
        return getJdbcTemplate()
                .queryForOption(queries.getTs(FIND_BY_BROADCAST_KEY), (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public ConferenceDto addLinkToCalendarEvent(ConferenceDto conference, String eventId) {
        MapF<String, Object> params = Cf.map(
                "event_id", eventId,
                "id", conference.getId()
        );
        params = params.plus1("id", conference.getId());
        if (getJdbcTemplate().update(queries.getTs(LINK_TO_CALENDAR_EVENT), params) > 0) {
            return conference.withCalendarEventId(eventId);
        }
        return conference;
    }

    @Override
    public SetF<ApiVersion> getConferencePeersVersions(String conferenceId) {
        MapF<String, ?> params = Cf.map("conf_id", conferenceId);
        return getJdbcTemplate().query(queries.getTs(GET_PEERS_VERSIONS),
                (rs, rowNum) -> ApiVersion.valueOf(rs.getString("api_version")), params).unique();
    }

    @Override
    protected String getTableName() {
        return "telemost.conferences";
    }

    @Override
    protected ListF<String> getFieldsToSelect() {
        return FIELDS_TO_SELECT;
    }

    @Override
    protected ConferenceDto parseRow(ResultSet resultSet) {
        try {
            return new ConferenceDto(Option.of(UUID.fromString(resultSet.getString("id"))),
                    resultSet.getString("conference_id"),
                    resultSet.getString("conference_password"),
                    resultSet.getString("limit_type"),
                    new Instant(resultSet.getTimestamp("created_at")),
                    resultSet.getString("short_url_id"),
                    Option.ofNullable(resultSet.getString("owner_uid")),
                    resultSet.getBoolean("staff_only"),
                    resultSet.getBoolean("is_permanent"),
                    resultSet.getBoolean("is_yateam"),
                    Option.ofNullable(resultSet.getString("event_id")),
                    Option.empty(),  // event caption
                    Option.empty(),  // event description
                    Option.empty(),  // start event
                    Option.empty()); // end event
        } catch (SQLException e) {
            throw new TelemostRuntimeException(e);
        }
    }
}
