package ru.yandex.chemodan.app.telemost.repository.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.exceptions.TelemostRuntimeException;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferencePeerDao;
import ru.yandex.chemodan.app.telemost.repository.model.ApiVersion;
import ru.yandex.chemodan.app.telemost.repository.model.ConferencePeerDto;
import ru.yandex.chemodan.app.telemost.repository.model.MediaSessionDto;
import ru.yandex.chemodan.app.telemost.repository.model.UserByConferences;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class ConferencePeerPgDaoImpl extends AbstractPgUUIDKeyDao<ConferencePeerDto> implements ConferencePeerDao {

    private static final ListF<String> FIELDS_TO_SELECT =
            Cf.list("id", "conference_id", "user_id", "uid", "display_name", "api_version", "peer_token");

    private static final String MERGE_BY_USER_ID = "merge-by-user_id";
    private static final String FIND_BY_CONFERENCE_AND_UID = "find-by-conference-and-uid";
    private static final String FIND_BY_CONFERENCE_AND_USERS_IDS = "find-by-conference-and-users-ids";
    private static final String FIND_ACTIVE_USERS = "find-active-users";
    private static final String UPDATE_BY_ID = "update-by-id";
    private static final String FIND_ACTIVE_OR_MVP1_USERS = "find-active-or-mvp1-users";
    private static final String FIND_ACTIVE_MVP1_USER = "find-active-mvp1-user";
    private static final String FIND_USER_BY_CONFERENCE_UID_CLIENT_INSTANCE_ID = "find-by-conference-uid-client_instance_id";
    private static final String FIND_USERS_AND_CONFERENCES_BY_UID = "find-users-and-conferences-by-uid";

    private final MapF<String, String> queries = Cf.map(
            MERGE_BY_USER_ID,
            "INSERT INTO " + getTableName() + " (user_id, uid, conference_id, display_name, api_version, peer_token) " +
                    "VALUES (:user_id, :uid, :conference_id, :display_name, :api_version, :peer_token) " +
                    "ON CONFLICT(conference_id, user_id) DO UPDATE SET display_name = :display_name, api_version = :api_version " +
                    "RETURNING *",
            FIND_BY_CONFERENCE_AND_UID, "SELECT " + getFieldsToSelect().map(field -> "u." + field).mkString(",") +
                    " FROM " + getTableName() + " AS u JOIN telemost.conferences AS c ON u.conference_id = c.id" +
                    " WHERE c.conference_id = :conference_id AND u.uid = :uid",
            FIND_BY_CONFERENCE_AND_USERS_IDS, "SELECT " + getFieldsToSelect().map(field -> "u." + field).mkString(",") +
                    " FROM " + getTableName() + " AS u JOIN telemost.conferences AS c ON u.conference_id = c.id" +
                    " WHERE c.conference_id = :conference_id AND u.user_id in ( :users_ids )"
    ).plus1(
            FIND_ACTIVE_USERS, "SELECT " + getFieldsToSelect().map(field -> "u." + field).mkString(",") +
                    " FROM " + getTableName() + " AS u JOIN telemost.conferences AS c ON u.conference_id = c.id" +
                    " JOIN telemost.media_sessions AS ms ON u.id = ms.user_id" +
                    " WHERE c.conference_id = :conference_id AND disconnected_at = :active_session_ts"
    ).plus1(
            UPDATE_BY_ID, "UPDATE " + getTableName() + " SET display_name = :display_name, api_version = :api_version " +
                    "WHERE id = :id RETURNING *"
    ).plus1(
            FIND_ACTIVE_OR_MVP1_USERS, "SELECT " + getFieldsToSelect().map(field -> "u." + field).mkString(",") +
                    " FROM " + getTableName() + " AS u JOIN telemost.conferences AS c ON u.conference_id = c.id" +
                    " LEFT JOIN telemost.media_sessions AS ms ON u.id = ms.user_id" +
                    " WHERE c.conference_id = :conference_id AND disconnected_at = :active_session_ts"
    ).plus1(
            FIND_ACTIVE_MVP1_USER, "SELECT " + getFieldsToSelect().map(field -> "u." + field).mkString(",") +
                    " FROM " + getTableName() + " AS u" +
                    " LEFT JOIN telemost.media_sessions AS ms ON u.id = ms.user_id" +
                    " WHERE u.conference_id = :conference_id AND u.user_id = :user_id AND disconnected_at = :active_session_ts"
    ).plus1(FIND_USER_BY_CONFERENCE_UID_CLIENT_INSTANCE_ID, "SELECT * " +
            " FROM " + getTableName() + " AS u" +
            " WHERE conference_id = :conference_id AND EXISTS (SELECT 1 " +
                                               "FROM telemost.media_sessions AS ms " +
                                               "WHERE u.id = ms.user_id AND ms.client_instance_id = :client_instance_id) " +
                     " AND uid %s " +
            " ORDER BY id LIMIT 1"
    ).plus1(FIND_USERS_AND_CONFERENCES_BY_UID, "SELECT c.conference_id AS conference," + getFieldsToSelect().map(field -> "u." + field).mkString(",") +
            " FROM " + getTableName() + " AS u JOIN telemost.conferences as c ON u.conference_id = c.id " +
            " WHERE u.uid = :uid");

    public ConferencePeerPgDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public ConferencePeerDto mergeByPeerId(ConferencePeerDto userDto) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("user_id", userDto.getPeerId());
        params.put("uid", userDto.getUid().getOrNull());
        params.put("conference_id", userDto.getConferenceId());
        params.put("display_name", userDto.getDisplayName());
        params.put("api_version", userDto.getApiVersion().map(ApiVersion::name).getOrNull());
        params.put("peer_token", userDto.getPeerToken().getOrNull());

        return getJdbcTemplate().query(queries.getTs(MERGE_BY_USER_ID), (rs, rowNum) -> parseRow(rs), params).first();
    }

    @Override
    public ListF<ConferencePeerDto> findPeersInConference(String conferenceId, ListF<String> userIds) {
        MapF<String, ?> params = Cf.map("conference_id", conferenceId,
                "users_ids", userIds);
        return getJdbcTemplate().query(queries.getTs(FIND_BY_CONFERENCE_AND_USERS_IDS),
                (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public ListF<ConferencePeerDto> findActivePeers(String conferenceId) {
        MapF<String, ?> params = Cf.map("conference_id", conferenceId,
                "active_session_ts", MediaSessionDto.ACTIVE_SESSION_INSTANT);
        return getJdbcTemplate().query(queries.getTs(FIND_ACTIVE_USERS),
                (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public ListF<ConferencePeerDto> findActiveOrMVP1Peers(String conferenceId) {
        MapF<String, ?> params = Cf.map("conference_id", conferenceId,
                "active_session_ts", MediaSessionDto.ACTIVE_SESSION_INSTANT);
        return getJdbcTemplate().query(queries.getTs(FIND_ACTIVE_OR_MVP1_USERS),
                (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public Option<ConferencePeerDto> findActiveMVP1Peer(UUID conferenceId, String peerId) {
        MapF<String, ?> params = Cf.map("conference_id", conferenceId,
                "user_id", peerId,
                "active_session_ts", MediaSessionDto.ACTIVE_SESSION_INSTANT);
        return getJdbcTemplate().queryForOption(queries.getTs(FIND_ACTIVE_MVP1_USER),
                (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public ConferencePeerDto updateById(UUID id, String displayName, ApiVersion apiVersion) {
        MapF<String, ?> params = Cf.map("id", id,
                "display_name", displayName,
                "api_version", apiVersion.name());
        return getJdbcTemplate().queryForOption(queries.getTs(UPDATE_BY_ID),
                (rs, rowNum) -> parseRow(rs), params).getOrThrow(IllegalStateException::new);
    }

    @Override
    public Option<ConferencePeerDto> findUserByConferenceAndUidAndClientInstanceId(UUID conferenceId, String clientInstanceId, Option<String> uid) {
        MapF<String, Object> params = Cf.map("conference_id", conferenceId,
                "client_instance_id", clientInstanceId);
        String query = queries.getTs(FIND_USER_BY_CONFERENCE_UID_CLIENT_INSTANCE_ID);
        if (uid.isPresent()) {
            params = params.plus1("uid", uid.get());
            query = String.format(query, "= :uid");
        } else {
            query = String.format(query, "IS NULL");
        }
        return getJdbcTemplate().queryForOption(query, (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public ListF<ConferencePeerDto> findByConferenceAndUid(String conferenceId, PassportOrYaTeamUid uid) {
        MapF<String, ?> params = Cf.map("conference_id", conferenceId,
                "uid", uid.asString());
        return getJdbcTemplate().query(queries.getTs(FIND_BY_CONFERENCE_AND_UID), (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public ListF<UserByConferences> findByUid(String uid) {
        MapF<String, ?> params = Cf.map("uid", uid);
        return getJdbcTemplate().query(queries.getTs(FIND_USERS_AND_CONFERENCES_BY_UID),
                (rs, rowNum) -> parseRowConferences(rs), params);
    }

    @Override
    protected String getTableName() {
        return "telemost.user_data";
    }

    @Override
    protected ListF<String> getFieldsToSelect() {
        return FIELDS_TO_SELECT;
    }

    @Override
    protected ConferencePeerDto parseRow(ResultSet resultSet) {
        try {
            return new ConferencePeerDto(
                    Option.of(UUID.fromString(resultSet.getString("id"))),
                    resultSet.getString("user_id"),
                    Option.ofNullable(resultSet.getString("uid")),
                    UUID.fromString(resultSet.getString("conference_id")),
                    resultSet.getString("display_name"),
                    Option.ofNullable(resultSet.getString("api_version")).map(ApiVersion::valueOf),
                    Option.ofNullable(resultSet.getString("peer_token"))
            );
        } catch (SQLException e) {
            throw new TelemostRuntimeException(e);
        }
    }

    protected UserByConferences parseRowConferences(ResultSet resultSet) {
        try {
            return new UserByConferences(
                    new ConferencePeerDto(
                            Option.of(UUID.fromString(resultSet.getString("id"))),
                            resultSet.getString("user_id"),
                            Option.ofNullable(resultSet.getString("uid")),
                            UUID.fromString(resultSet.getString("conference_id")),
                            resultSet.getString("display_name"),
                            Option.ofNullable(resultSet.getString("api_version")).map(ApiVersion::valueOf),
                            Option.ofNullable(resultSet.getString("peer_token"))
                    ),
                    resultSet.getString("conference")
            );
        } catch (SQLException e) {
            throw new TelemostRuntimeException(e);
        }
    }
}
