package ru.yandex.chemodan.app.telemost.repository.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import lombok.SneakyThrows;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.telemost.exceptions.TelemostRuntimeException;
import ru.yandex.chemodan.app.telemost.repository.dao.BroadcastDao;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceStateDao;
import ru.yandex.chemodan.app.telemost.repository.dao.StreamDao;
import ru.yandex.chemodan.app.telemost.repository.model.BroadcastDto;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceStateDto;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceStateNode;
import ru.yandex.chemodan.app.telemost.repository.model.StreamDto;
import ru.yandex.chemodan.app.telemost.services.model.ConferenceState;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class ConferenceStatePgDaoImpl extends AbstractPgUUIDKeyDao<ConferenceStateDto> implements ConferenceStateDao {
    private static final String TABLE_NAME = "telemost.conference_state";

    private static final String ID_FIELD = "id";
    private static final String CONFERENCE_ID_FIELD = "conference_id";
    private static final String VERSION_FIELD = "version";
    private static final String STATE_FIELD = "state";
    private static final String BROADCAST_ALLOWED_FIELD = "broadcast_allowed";
    private static final String BROADCAST_FEATURE_ENABLED_FIELD = "broadcast_enabled";

    private static final ListF<String> FIELDS_TO_INSERT =
            Cf.list(CONFERENCE_ID_FIELD, VERSION_FIELD, STATE_FIELD, BROADCAST_ALLOWED_FIELD);
    private static final ListF<String> FIELDS_TO_SELECT = Cf.list(ID_FIELD).plus(FIELDS_TO_INSERT);

    private static final String SQL_MERGE = "merge";
    private static final String SQL_FIND_STATE_BY_CONFERENCE_ID = "find-state-by-conference-id";
    private static final String SQL_INCREMENT_VERSION = "increment-version";

    private static final Function<String, String>
            jsonFieldMapper = field -> !STATE_FIELD.equals(field) ? field : (STATE_FIELD + "::text");

    private final MapF<String, String> queries = Cf.map(
            SQL_MERGE, "WITH cs AS (" +
                    "INSERT INTO " + TABLE_NAME + " AS t (" + FIELDS_TO_INSERT.mkString(", ") + ") " +
                    "VALUES (:conference_id, 0, :state::jsonb, :broadcast_allowed) " +
                    "ON CONFLICT (conference_id) DO UPDATE SET" +
                    " version = t.version + 1," +
                    " state = :state::jsonb," +
                    " broadcast_allowed = :broadcast_allowed," +
                    " updated_at = now() " +
                    "RETURNING *) " +
                    "SELECT " + FIELDS_TO_SELECT.map(jsonFieldMapper).map(field -> "cs." + field).mkString(", ") +
                    ", u." + BROADCAST_FEATURE_ENABLED_FIELD +
                    " FROM cs" +
                    " LEFT JOIN telemost.conference_users cu ON cu.conference_id = cs.conference_id" +
                    "     AND cu.role = 'OWNER'" +
                    " LEFT JOIN telemost.users u ON cu.uid = u.uid" +
                    " WHERE cs.conference_id = :conference_id",
            SQL_FIND_STATE_BY_CONFERENCE_ID,
            "SELECT " + FIELDS_TO_SELECT.map(jsonFieldMapper).map(field -> "cs." + field).mkString(", ") +
                    ", u." + BROADCAST_FEATURE_ENABLED_FIELD +
                    " FROM " + TABLE_NAME + " cs" +
                    " LEFT JOIN telemost.conference_users cu ON cu.conference_id = cs.conference_id" +
                    "     AND cu.role = 'OWNER'" +
                    " LEFT JOIN telemost.users u ON cu.uid = u.uid" +
                    " WHERE cs.conference_id = :conference_id",
            SQL_INCREMENT_VERSION,
            "WITH cs AS (" +
                    "UPDATE " + TABLE_NAME + " SET version = version + 1" +
                    " WHERE " + CONFERENCE_ID_FIELD + " = :conference_id" +
                    " RETURNING *) " +
                    "SELECT " + FIELDS_TO_SELECT.map(jsonFieldMapper).map(field -> "cs." + field).mkString(", ") +
                    ", u." + BROADCAST_FEATURE_ENABLED_FIELD +
                    " FROM cs" +
                    " LEFT JOIN telemost.conference_users cu ON cu.conference_id = cs.conference_id" +
                    "     AND cu.role = 'OWNER'" +
                    " LEFT JOIN telemost.users u ON cu.uid = u.uid" +
                    " WHERE cs.conference_id = :conference_id"
    );

    private final BroadcastDao broadcastDao;
    private final StreamDao streamDao;

    public ConferenceStatePgDaoImpl(JdbcTemplate3 jdbcTemplate, BroadcastDao broadcastDao,
                                       StreamDao streamDao) {
        super(jdbcTemplate);

        this.broadcastDao = broadcastDao;
        this.streamDao = streamDao;
    }

    @Override
    @SneakyThrows
    public ConferenceStateDto updateVersion(ConferenceStateDto state) {
        MapF<String, Object> params = Cf.map(
                CONFERENCE_ID_FIELD, state.getConferenceId(),
                STATE_FIELD, (new ConferenceStateNode(state)).asString(),
                BROADCAST_ALLOWED_FIELD, state.isBroadcastAllowed()
        );
        return getJdbcTemplate().query(
                queries.getTs(SQL_MERGE),
                (rs, rowNum) -> parseRow(rs),
                params
        ).first();
    }

    @Override
    public ConferenceStateDto incrementVersion(UUID conferenceId) {
        MapF<String, Object> params = Cf.map(
                CONFERENCE_ID_FIELD, conferenceId
        );
        return getJdbcTemplate().query(
                queries.getTs(SQL_INCREMENT_VERSION),
                (rs, rowNum) -> parseRow(rs),
                params
        ).first();
    }

    @Override
    public Option<ConferenceStateDto> findState(UUID conferenceId) {
        return getJdbcTemplate().query(queries.getTs(SQL_FIND_STATE_BY_CONFERENCE_ID),
                (rs, rowNum) -> parseRow(rs),
                Cf.map(CONFERENCE_ID_FIELD, conferenceId)).firstO();
    }

    @Override
    public ConferenceState from(ConferenceStateDto conferenceStateDto) {
        return new ConferenceState(conferenceStateDto);
    }

    @Override
    protected String getTableName() {
        return TABLE_NAME;
    }

    @Override
    protected ListF<String> getFieldsToSelect() {
        return FIELDS_TO_SELECT;
    }

    @Override
    protected ConferenceStateDto parseRow(ResultSet rs) {
        try {
            UUID conferenceId = UUID.fromString(rs.getString(CONFERENCE_ID_FIELD));
            Option<BroadcastDto> broadcast = broadcastDao.findByConferenceId(conferenceId);
            Option<StreamDto> stream = Option.empty();
            if (broadcast.isPresent()) {
                stream = streamDao.findActiveByBroadcastKey(broadcast.get().getBroadcastKey());
            }

            ConferenceStateNode node = ConferenceStateNode.fromString(rs.getString(STATE_FIELD));

            return new ConferenceStateDto(
                    Option.of(UUID.fromString(rs.getString(ID_FIELD))),
                    conferenceId,
                    rs.getLong(VERSION_FIELD),
                    node.isLocalRecordingAllowed(),
                    node.isCloudRecordingAllowed(),
                    node.isChatAllowed(),
                    node.isControlAllowed(),
                    rs.getBoolean(BROADCAST_ALLOWED_FIELD),
                    node.getChatPath(),
                    broadcast,
                    stream,
                    rs.getBoolean(BROADCAST_FEATURE_ENABLED_FIELD)
            );
        } catch (SQLException e) {
            throw new TelemostRuntimeException(e);
        }
    }
}
