package ru.yandex.chemodan.app.telemost.repository.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;
import java.util.stream.Collectors;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.telemost.exceptions.TelemostRuntimeException;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceUserDao;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceUserDto;
import ru.yandex.chemodan.app.telemost.repository.model.UserRole;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

import static ru.yandex.bolts.collection.Tuple2.tuple;

public class ConferenceUserPgDaoImpl extends AbstractPgUUIDKeyDao<ConferenceUserDto> implements ConferenceUserDao {

    private static final String ID_FIELD = "id";
    private static final String UID_FIELD = "uid";
    private static final String CONFERENCE_ID_FIELD = "conference_id";
    private static final String ROLE_FIELD = "role";
    private static final ListF<String> FIELDS_TO_INSERT = Cf.list(UID_FIELD, CONFERENCE_ID_FIELD, ROLE_FIELD);
    private static final ListF<String> FIELDS_TO_SELECT = Cf.list(ID_FIELD, UID_FIELD, CONFERENCE_ID_FIELD, ROLE_FIELD);

    private static final String TABLE_NAME = "telemost.conference_users";

    public ConferenceUserPgDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public void insertIfNotExists(UUID conferenceId, PassportOrYaTeamUid uid, UserRole role){
        getJdbcTemplate().update(Queries.INSERT_IF_NOT_EXISTS.query,
                Cf.map(CONFERENCE_ID_FIELD, conferenceId, UID_FIELD, uid.asString(), ROLE_FIELD, role.name()));
    }

    @Override
    public ConferenceUserDto upsert(UUID conferenceId, PassportOrYaTeamUid uid, UserRole role){
        return getJdbcTemplate().query(Queries.UPSERT.query, (rs, rowNum) -> parseRow(rs),
                Cf.map(CONFERENCE_ID_FIELD, conferenceId, UID_FIELD, uid.asString(), ROLE_FIELD, role.name())).get(0);
    }

    @Override
    public ListF<ConferenceUserDto> findByConference(UUID conferenceId) {
        return getJdbcTemplate().query(Queries.FIND_BY_CONFERENCE_ID.query,
                (rs, rowNum) -> parseRow(rs),
                Cf.map(CONFERENCE_ID_FIELD, conferenceId));
    }

    @Override
    public ListF<ConferenceUserDto> findByConferenceAndRole(UUID conferenceId, UserRole role) {
        return getJdbcTemplate().query(Queries.FIND_BY_CONFERENCE_ID_AND_ROLE.query,
                (rs, rowNum) -> parseRow(rs),
                Cf.map(CONFERENCE_ID_FIELD, conferenceId, ROLE_FIELD, role.name()));
    }

    @Override
    public Option<ConferenceUserDto> findByConferenceAndUid(UUID conferenceId, PassportOrYaTeamUid uid) {
        return getJdbcTemplate().query(Queries.FIND_BY_CONFERENCE_ID_AND_UID.query,
                (rs, rowNum) -> parseRow(rs),
                Cf.map(CONFERENCE_ID_FIELD, conferenceId, UID_FIELD, uid.asString())).firstO();
    }

    @Override
    public Option<ConferenceUserDto> findOwner(UUID conferenceId) {
        return getJdbcTemplate().queryForOption(Queries.FIND_CONFERENCE_OWNER.query, (rs, rowNum) -> parseRow(rs),
                Cf.map(CONFERENCE_ID_FIELD, conferenceId, ROLE_FIELD, UserRole.OWNER.name()));
    }

    @Override
    public MapF<String, ConferenceUserDto> findByConferenceAndPeers(UUID conferenceId, ListF<String> peerIds) {
        if (peerIds.isEmpty()) {
            return Cf.map();
        }
        return getJdbcTemplate().query(Queries.FIND_BY_CONFERENCE_ID_AND_PEERS.query,
                (rs, rowNum) -> tuple(rs.getString("peer_id"), parseRow(rs)),
                Cf.map("peer_ids", peerIds, "conference_id", conferenceId))
                .toMap(Tuple2::get1, Tuple2::get2);
    }

    @Override
    protected String getTableName() {
        return TABLE_NAME;
    }

    @Override
    protected ListF<String> getFieldsToSelect() {
        return FIELDS_TO_SELECT;
    }

    @Override
    protected ConferenceUserDto parseRow(ResultSet rs) {
        try {
            return new ConferenceUserDto(
                    Option.of(UUID.fromString(rs.getString(ID_FIELD))),
                    UUID.fromString(rs.getString(CONFERENCE_ID_FIELD)),
                    rs.getString(UID_FIELD),
                    UserRole.valueOf(rs.getString(ROLE_FIELD))
            );
        } catch (SQLException e) {
            throw new TelemostRuntimeException(e);
        }
    }

    private enum Queries {
        FIND_BY_CONFERENCE_ID("SELECT " + FIELDS_TO_SELECT.mkString(", ") +
                        " FROM " + TABLE_NAME +
                        " WHERE conference_id = :conference_id"),
        FIND_CONFERENCE_OWNER(FIND_BY_CONFERENCE_ID.query + "" +
                                " AND role = '" + UserRole.OWNER.name() + "'"),
        FIND_BY_CONFERENCE_ID_AND_ROLE(FIND_BY_CONFERENCE_ID.query + "" +
                " AND role = :role"),
        INSERT("INSERT INTO " + TABLE_NAME + " (" + FIELDS_TO_INSERT.mkString(",") + ")"+
                        " VALUES (" + FIELDS_TO_INSERT.stream().map(str -> ":"+str).collect(Collectors.joining(",")) + ")"),
        UPSERT(INSERT.query +
                " ON CONFLICT(conference_id, uid) DO UPDATE SET role = :role" +
                " RETURNING *"),
        INSERT_IF_NOT_EXISTS(INSERT.query +
                " ON CONFLICT(conference_id, uid) DO NOTHING"),
        FIND_BY_CONFERENCE_ID_AND_UID(
                "SELECT " + FIELDS_TO_SELECT.mkString(", ") +
                        " FROM " + TABLE_NAME +
                        " WHERE conference_id = :conference_id" +
                        " AND uid = :uid"),
        FIND_BY_CONFERENCE_ID_AND_PEERS("SELECT ud.user_id AS peer_id, "
                + FIELDS_TO_SELECT.map(field -> "cu."+field).mkString(", ") +
                " FROM " + TABLE_NAME + " cu" +
                " JOIN telemost.user_data ud ON cu.conference_id = ud.conference_id AND cu.uid = ud.uid" +
                " WHERE ud.conference_id = :conference_id" +
                " AND ud.user_id IN ( :peer_ids )");

        private final String query;

        Queries(String query) {
            this.query = query;
        }
    }
}
