package ru.yandex.chemodan.app.telemost.repository.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.exceptions.TelemostRuntimeException;
import ru.yandex.chemodan.app.telemost.repository.dao.MediaSessionDao;
import ru.yandex.chemodan.app.telemost.repository.model.MediaSessionDto;
import ru.yandex.chemodan.app.telemost.repository.model.MediaSessionWithPeerTokenDto;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class MediaSessionPgDaoImpl extends AbstractPgUUIDKeyDao<MediaSessionDto> implements MediaSessionDao {

    private static final ListF<String> FIELDS_TO_SELECT =
            Cf.list("id", "media_session_id", "user_id", "disconnected_at", "client_instance_id");

    private static final String DEACTIVATE_USER_SESSION = "deactivate-user-session";
    private static final String GET_ACTIVE_SESSION = "get-active-session";
    private static final String DEACTIVATE_SESSION = "deactivate-session";
    private static final String INSERT_ONE = "insert-one";
    private static final String GET_SESSION_WITH_TOKEN = "get-session-with-token";
    private static final String TRY_TO_INSERT = "try-to-insert";

    private final MapF<String, String> queries = Cf.map(
            DEACTIVATE_USER_SESSION, "UPDATE " + getTableName() + " SET disconnected_at = now() " +
                    "WHERE user_id IN (:user_ids) AND (disconnected_at IS NULL OR disconnected_at = :active_session_ts) ",
            GET_ACTIVE_SESSION, "SELECT " + getFieldsToSelect().map(field -> "ms." + field).mkString(", ") +
                    " FROM " + getTableName() + " AS ms JOIN telemost.user_data AS u ON u.id = ms.user_id" +
                    " WHERE u.conference_id = :conference_id AND disconnected_at = :active_session_ts " +
                    "AND ms.client_instance_id = :client_instance_id",
            DEACTIVATE_SESSION, "UPDATE " + getTableName() + " SET disconnected_at = now() " +
                    "WHERE id = :id AND disconnected_at = :active_session_ts"
    ).plus1(GET_SESSION_WITH_TOKEN, "SELECT " + getFieldsToSelect().map(field -> "ms." + field).mkString(", ") + ", u.peer_token" +
            " FROM " + getTableName() + " AS ms JOIN telemost.user_data AS u ON u.id = ms.user_id" +
            " WHERE u.conference_id = :conference_id AND u.user_id = :user_id " +
            "AND ms.media_session_id = :media_session_id"
    ).plus1(TRY_TO_INSERT, "INSERT INTO " + getTableName() + " (media_session_id, disconnected_at, user_id, client_instance_id) " +
            "VALUES (:media_session_id, :disconnected_at, :user_id, :client_instance_id) " +
            "ON CONFLICT (user_id, media_session_id) DO NOTHING " +
            "RETURNING *"
    ).plus1(INSERT_ONE, "INSERT INTO " + getTableName() + " (media_session_id, disconnected_at, user_id, client_instance_id) " +
            "VALUES (:media_session_id, :disconnected_at, :user_id, :client_instance_id) " +
            "ON CONFLICT (user_id, disconnected_at) DO NOTHING " +
            "RETURNING *");

    public MediaSessionPgDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public int deactivateUsers(ListF<UUID> userIds) {
        MapF<String, ?> params = Cf.map("user_ids", userIds,
                "active_session_ts", MediaSessionDto.ACTIVE_SESSION_INSTANT);
        return getJdbcTemplate().update(queries.getTs(DEACTIVATE_USER_SESSION), params);
    }

    @Override
    public Option<MediaSessionDto> getActiveMediaSession(UUID conferenceId, String clientInstanceId, Option<String> uid) {
        MapF<String, ?> params = Cf.map("conference_id", conferenceId,
                "client_instance_id", clientInstanceId,
                "uid", uid.getOrNull(),
                "active_session_ts", MediaSessionDto.ACTIVE_SESSION_INSTANT);
        String uidCondition = uid.isPresent() ? " AND uid = :uid" : " AND uid IS NULL";
        String query = queries.getTs(GET_ACTIVE_SESSION) + uidCondition;
        return getJdbcTemplate().queryForOption(query, (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public int deactivateSession(MediaSessionDto mediaSessionDto) {
        MapF<String, ?> params = Cf.map("id", mediaSessionDto.getId(),
                "active_session_ts", MediaSessionDto.ACTIVE_SESSION_INSTANT);
        return getJdbcTemplate().update(queries.getTs(DEACTIVATE_SESSION), params);
    }

    @Override
    public Option<MediaSessionDto> insert(MediaSessionDto mediaSessionDto) {
        MapF<String, ?> params = Cf.map("media_session_id", mediaSessionDto.getMediaSessionId(),
                "user_id", mediaSessionDto.getUserId(),
                "disconnected_at", mediaSessionDto.getDisconnectedAt(),
                "client_instance_id", mediaSessionDto.getClientInstanceId().getOrNull());
        return getJdbcTemplate().queryForOption(queries.getTs(INSERT_ONE), (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public Option<MediaSessionDto> tryToInsert(MediaSessionDto mediaSessionDto) {
        MapF<String, ?> params = Cf.map("media_session_id", mediaSessionDto.getMediaSessionId(),
                "user_id", mediaSessionDto.getUserId(),
                "disconnected_at", mediaSessionDto.getDisconnectedAt(),
                "client_instance_id", mediaSessionDto.getClientInstanceId().getOrNull());
        return getJdbcTemplate().queryForOption(queries.getTs(TRY_TO_INSERT), (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public Option<MediaSessionWithPeerTokenDto> getMediaSessionForUserAndConference(UUID conferenceId,
            String userId, String mediaSessionId)
    {
        MapF<String, ?> params = Cf.map("conference_id", conferenceId,
                "user_id", userId,
                "media_session_id", mediaSessionId);
        return getJdbcTemplate().queryForOption(queries.getTs(GET_SESSION_WITH_TOKEN),
                (rs, rowNum) -> parseMediaSessionWithPeer(rs), params);
    }

    @Override
    protected String getTableName() {
        return "telemost.media_sessions";
    }

    @Override
    protected ListF<String> getFieldsToSelect() {
        return FIELDS_TO_SELECT;
    }

    @Override
    protected MediaSessionDto parseRow(ResultSet resultSet) {
        try {
            return new MediaSessionDto(
                    Option.of(UUID.fromString(resultSet.getString("id"))),
                    resultSet.getString("media_session_id"),
                    UUID.fromString(resultSet.getString("user_id")),
                    new Instant(resultSet.getTimestamp("disconnected_at")),
                    Option.of(resultSet.getString("client_instance_id"))
            );
        } catch (SQLException e) {
            throw new TelemostRuntimeException(e);
        }
    }

    private MediaSessionWithPeerTokenDto parseMediaSessionWithPeer(ResultSet resultSet) {
        try {
            return new MediaSessionWithPeerTokenDto(
                    parseRow(resultSet),
                    resultSet.getString("peer_token"));
        } catch (SQLException e) {
            throw new TelemostRuntimeException(e);
        }
    }
}
