package ru.yandex.chemodan.app.telemost.repository.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;
import java.util.stream.Collectors;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.telemost.exceptions.TelemostRuntimeException;
import ru.yandex.chemodan.app.telemost.repository.dao.StreamDao;
import ru.yandex.chemodan.app.telemost.repository.model.StreamDto;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class StreamPgDaoImpl extends AbstractPgUUIDKeyDao<StreamDto> implements StreamDao {

    private static final String TABLE_NAME = "telemost.streams";

    private static final String ID_FIELD = "id";
    private static final String BROADCAST_ID_FIELD = "broadcast_id";
    private static final String UGC_LIVE_SLUG_FIELD = "ugc_live_slug";
    private static final String OWNER_UID_FIELD = "owner_uid";
    private static final String STARTED_AT_FIELD = "started_at";
    private static final String STOPPED_AT_FIELD = "stopped_at";
    private static final String TRANSLATOR_FIELD = "translator";
    private static final String TRANSLATOR_TOKEN_FIELD = "translator_token";
    private static final String RTMP_KEY_FIELD = "rtmp_key";
    private static final String TRANSLATOR_LAST_KEEP_ALIVE_FIELD = "translator_last_keep_alive";
    private static final String TRANSLATOR_PEER_ID_FIELD = "translator_peer_id";

    private static final ListF<String> FIELDS_TO_UPDATE = Cf.list(OWNER_UID_FIELD, STARTED_AT_FIELD, STOPPED_AT_FIELD,
            TRANSLATOR_FIELD, TRANSLATOR_TOKEN_FIELD, RTMP_KEY_FIELD, TRANSLATOR_LAST_KEEP_ALIVE_FIELD,
            TRANSLATOR_PEER_ID_FIELD);
    private static final ListF<String> FIELDS_TO_INSERT = Cf.list(ID_FIELD, BROADCAST_ID_FIELD, UGC_LIVE_SLUG_FIELD)
            .plus(FIELDS_TO_UPDATE);
    private static final ListF<String> FIELDS_TO_SELECT = FIELDS_TO_INSERT;

    private final String playerBaseUrl;

    public StreamPgDaoImpl(JdbcTemplate3 jdbcTemplate, String playerBaseUrl) {
        super(jdbcTemplate);

        this.playerBaseUrl = playerBaseUrl;
    }

    @Override
    public String getStreamUri(String ugcLiveSlug) {
        return playerBaseUrl + ugcLiveSlug;
    }

    @Override
    public Option<StreamDto> findActiveByBroadcastKey(String broadcastKey) {
        return getJdbcTemplate().query(StreamPgDaoImpl.Queries.SELECT_ACTIVE_BY_BROADCAST_KEY.query, (rs, rowNum) -> parseRow(rs),
                Cf.map("broadcast_key", broadcastKey)).firstO();
    }

    @Override
    public StreamDto insert(StreamDto stream) {
        MapF<String, Object> params = Cf.toMap(Tuple2List.fromPairs(
                ID_FIELD, stream.getId(),
                BROADCAST_ID_FIELD, stream.getBroadcastId(),
                UGC_LIVE_SLUG_FIELD, stream.getUgcLiveSlug(),
                OWNER_UID_FIELD, stream.getOwnerUid().orElse((String) null),
                STARTED_AT_FIELD, stream.getStartedAt().orElse((Instant) null)));
        params = params
                .plus1(STOPPED_AT_FIELD, stream.getStoppedAt().orElse((Instant) null))
                .plus1(TRANSLATOR_FIELD, stream.getTranslator().orElse((String) null))
                .plus1(TRANSLATOR_TOKEN_FIELD, stream.getTranslatorToken().orElse((String) null))
                .plus1(RTMP_KEY_FIELD, stream.getRtmpKey().orElse((String) null))
                .plus1(TRANSLATOR_LAST_KEEP_ALIVE_FIELD, stream.getTranslatorLastKeepAlive().orElse((Instant) null))
                .plus1(TRANSLATOR_PEER_ID_FIELD, stream.getTranslatorPeerId().orElse((String) null));
        return getJdbcTemplate().query(StreamPgDaoImpl.Queries.INSERT.query, (rs, rowNum) -> parseRow(rs),
                params).get(0);
    }

    @Override
    public Option<StreamDto> update(StreamDto stream) {
        MapF<String, Object> params = Cf.map(
                ID_FIELD, stream.getId(),
                OWNER_UID_FIELD, stream.getOwnerUid(),
                STARTED_AT_FIELD, stream.getStartedAt().orElse((Instant) null),
                STOPPED_AT_FIELD, stream.getStoppedAt().orElse((Instant) null));
        params = params
                .plus1(TRANSLATOR_FIELD, stream.getTranslator().orElse((String) null))
                .plus1(TRANSLATOR_TOKEN_FIELD, stream.getTranslatorToken().orElse((String) null))
                .plus1(RTMP_KEY_FIELD, stream.getRtmpKey().orElse((String) null))
                .plus1(TRANSLATOR_LAST_KEEP_ALIVE_FIELD, stream.getTranslatorLastKeepAlive().orElse((Instant) null))
                .plus1(TRANSLATOR_PEER_ID_FIELD, stream.getTranslatorPeerId().orElse((String) null));
        return getJdbcTemplate().query(StreamPgDaoImpl.Queries.UPDATE.query, (rs, rowNum) -> parseRow(rs),
                params).firstO();
    }

    @Override
    public void delete(UUID streamId) {
        getJdbcTemplate().update(Queries.DELETE.query, Cf.map(ID_FIELD, streamId));
    }

    @Override
    public ListF<StreamDto> getActiveStreams() {
        return getJdbcTemplate().query(StreamPgDaoImpl.Queries.SELECT_ACTIVE_STREAMS.query, (rs, rowNum) -> parseRow(rs),
                Cf.map()).toList();
    }

    @Override
    protected String getTableName() {
        return TABLE_NAME;
    }

    @Override
    protected ListF<String> getFieldsToSelect() {
        return FIELDS_TO_SELECT;
    }

    @Override
    protected StreamDto parseRow(ResultSet rs) {
        try {
            return new StreamDto(
                    UUID.fromString(rs.getString(ID_FIELD)),
                    UUID.fromString(rs.getString(BROADCAST_ID_FIELD)),
                    rs.getString(UGC_LIVE_SLUG_FIELD),
                    Option.ofNullable(rs.getString(OWNER_UID_FIELD)),
                    Option.ofNullable(rs.getTimestamp(STARTED_AT_FIELD)).map(Instant::new),
                    Option.ofNullable(rs.getTimestamp(STOPPED_AT_FIELD)).map(Instant::new),
                    Option.ofNullable(rs.getString(TRANSLATOR_FIELD)),
                    Option.ofNullable(rs.getString(TRANSLATOR_TOKEN_FIELD)),
                    Option.ofNullable(rs.getString(RTMP_KEY_FIELD)),
                    Option.ofNullable(rs.getTimestamp(TRANSLATOR_LAST_KEEP_ALIVE_FIELD)).map(Instant::new),
                    Option.ofNullable(rs.getString(TRANSLATOR_PEER_ID_FIELD)),
                    getStreamUri(rs.getString(UGC_LIVE_SLUG_FIELD))
            );
        } catch (SQLException e) {
            throw new TelemostRuntimeException(e);
        }
    }

    private enum Queries {
        SELECT("SELECT " + FIELDS_TO_SELECT.mkString(",") + " FROM " + TABLE_NAME +
                " WHERE " + ID_FIELD + " = :id"),
        SELECT_ACTIVE_BY_BROADCAST_KEY("SELECT " + FIELDS_TO_SELECT.map("s."::concat).mkString(",") +
                " FROM " + TABLE_NAME + " AS s" +
                " JOIN telemost.broadcasts AS t" +
                " ON s." + BROADCAST_ID_FIELD + " = t.id" +
                " WHERE s." + STOPPED_AT_FIELD + " IS NULL" +
                " AND t.broadcast_key = :broadcast_key" +
                " ORDER BY " + STARTED_AT_FIELD + " DESC LIMIT 1"),
        INSERT("INSERT INTO " + TABLE_NAME + " (" + FIELDS_TO_INSERT.mkString(",") + ")" +
                " VALUES (" + FIELDS_TO_INSERT.stream().map(f -> ":" + f).collect(Collectors.joining(",")) + ")" +
                " RETURNING *"),
        UPDATE("UPDATE " + TABLE_NAME + " SET " + FIELDS_TO_UPDATE.stream().map(f -> f + "= :" + f).collect(Collectors.joining(",")) +
                " WHERE " + ID_FIELD + " = :id" +
                " RETURNING *"),
        DELETE("DELETE FROM " + TABLE_NAME + " WHERE " + ID_FIELD + " = :" + ID_FIELD),
        SELECT_ACTIVE_STREAMS("SELECT " + FIELDS_TO_SELECT.mkString(",") +
                " FROM " + TABLE_NAME +
                " WHERE " + STARTED_AT_FIELD + " IS NOT NULL AND " + STOPPED_AT_FIELD + " IS NULL");

        private final String query;

        Queries(String query) {
            this.query = query;
        }
    }
}
