package ru.yandex.chemodan.app.telemost.repository.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;
import java.util.stream.Collectors;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.exceptions.TelemostRuntimeException;
import ru.yandex.chemodan.app.telemost.repository.dao.UserDao;
import ru.yandex.chemodan.app.telemost.repository.model.UserDto;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class UserPgDaoImpl extends AbstractPgUidKeyDao<UserDto> implements UserDao {

    private static final String TABLE_NAME = "telemost.users";

    private static final String UID_FIELD = "uid";
    private static final String BROADCAST_ENABLED_FIELD = "broadcast_enabled";

    private static final ListF<String> FIELDS_TO_INSERT = Cf.list(UID_FIELD, BROADCAST_ENABLED_FIELD);
    private static final ListF<String> FIELDS_TO_SELECT = Cf.list(UID_FIELD, BROADCAST_ENABLED_FIELD);

    public UserPgDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public void insertIfNotExists(PassportOrYaTeamUid uid, boolean broadcastEnabled) {
        getJdbcTemplate().update(Queries.INSERT_IF_NOT_EXISTS.query,
                Cf.map(UID_FIELD, uid.asString(), BROADCAST_ENABLED_FIELD, broadcastEnabled));
    }

    @Override
    public UserDto upsert(PassportOrYaTeamUid uid, boolean broadcastEnabled) {
        return getJdbcTemplate().query(Queries.UPSERT.query, (rs, rowNum) -> parseRow(rs),
                Cf.map(UID_FIELD, uid.asString(), BROADCAST_ENABLED_FIELD, broadcastEnabled)).get(0);
    }

    @Override
    public Option<UserDto> findByUid(PassportOrYaTeamUid uid) {
        return getJdbcTemplate().query(Queries.SELECT.query, (rs, rowNum) -> parseRow(rs),
                Cf.map(UID_FIELD, uid.asString())).firstO();
    }

    @Override
    public Option<UserDto> findByConferenceOwner(UUID conferenceId) {
        return getJdbcTemplate().query(Queries.SELECT_BY_CONFERENCE_OWNER.query, (rs, rowNum) -> parseRow(rs),
                Cf.map("conference_id", conferenceId)).firstO();
    }

    @Override
    protected String getTableName() {
        return TABLE_NAME;
    }

    @Override
    protected ListF<String> getFieldsToSelect() {
        return FIELDS_TO_SELECT;
    }

    @Override
    protected UserDto parseRow(ResultSet rs) {
        try {
            return new UserDto(
                    rs.getString(UID_FIELD),
                    rs.getBoolean(BROADCAST_ENABLED_FIELD)
            );
        } catch (SQLException e) {
            throw new TelemostRuntimeException(e);
        }
    }

    private enum Queries {
        SELECT("SELECT " + FIELDS_TO_SELECT.mkString(",") + " FROM " + TABLE_NAME +
                " WHERE " + UID_FIELD + " = :uid"),
        SELECT_BY_CONFERENCE_OWNER("SELECT " + FIELDS_TO_SELECT.map(field -> "u." + field).mkString(", ") +
                " FROM " + TABLE_NAME + " u" +
                " JOIN telemost.conference_users cu" +
                " ON u.uid = cu.uid AND cu.role = 'OWNER'" +
                " WHERE cu.conference_id = :conference_id"),
        INSERT("INSERT INTO " + TABLE_NAME + " (" + FIELDS_TO_INSERT.mkString(",") + ")" +
                " VALUES (" + FIELDS_TO_INSERT.stream().map(str -> ":" + str).collect(Collectors.joining(",")) + ")"),
        UPSERT(INSERT.query +
                " ON CONFLICT(uid) DO UPDATE SET broadcast_enabled = :broadcast_enabled, updated_at = now()" +
                " RETURNING *"),
        INSERT_IF_NOT_EXISTS(INSERT.query +
                " ON CONFLICT(uid) DO NOTHING");

        private final String query;

        Queries(String query) {
            this.query = query;
        }
    }

}
