package ru.yandex.chemodan.app.telemost.repository.dao.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.exceptions.TelemostRuntimeException;
import ru.yandex.chemodan.app.telemost.repository.dao.UserTokenDtoDao;
import ru.yandex.chemodan.app.telemost.repository.model.UserTokenDto;
import ru.yandex.chemodan.app.telemost.repository.model.UserTokenStatus;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class UserTokenDtoPgDaoImpl extends AbstractPgUUIDKeyDao<UserTokenDto> implements UserTokenDtoDao {

    private static final ListF<String> FIELDS_TO_SELECT =
            Cf.list("id", "conference_id", "uid", "token", "created_at", "status");

    private static final String INSERT_ONE = "insert-one";

    private static final String ACTIVATE_TOKEN = "activate-token";

    private static final String EXPIRE_TOKEN = "expire-token";

    private static final String GET_TOKEN = "get-token";

    private static final String VALID_TOKEN_CONDITION = "((status = '%1$s' AND uid = :uid) OR " +
            "(status = '%2$s' AND EXTRACT(EPOCH FROM created_at) >= EXTRACT(EPOCH FROM now() - :expiration_seconds * INTERVAL '1 second') AND uid IS NULL))";

    private final MapF<String, String> queries = Cf.map(
            INSERT_ONE, "INSERT INTO " + getTableName() + " (conference_id, token, created_at, status) " +
                    "VALUES (:conference_id, :token, :created_at, :status) " +
                    "RETURNING *",
            ACTIVATE_TOKEN, String.format("UPDATE " + getTableName() + " SET status = '%1$s', uid = :uid " +
                    "WHERE conference_id = :conference_id AND token = :token " +
                    "AND " + VALID_TOKEN_CONDITION +
                    "RETURNING *", UserTokenStatus.ACTIVATED.name(), UserTokenStatus.CREATED.name()),
            EXPIRE_TOKEN, String.format("UPDATE " + getTableName() + " " +
                    "SET created_at = created_at - :expiration_seconds * INTERVAL '1 second'" +
                    "WHERE conference_id = :conference_id AND token = :token AND status = '%s' " +
                    "RETURNING *", UserTokenStatus.CREATED.name()),
            GET_TOKEN, String.format("SELECT * FROM " + getTableName() + " " +
                    "WHERE conference_id = :conference_id AND token = :token " +
                    "AND " + VALID_TOKEN_CONDITION, UserTokenStatus.ACTIVATED.name(), UserTokenStatus.CREATED.name()));

    public UserTokenDtoPgDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    protected String getTableName() {
        return "telemost.user_token";
    }

    @Override
    protected ListF<String> getFieldsToSelect() {
        return FIELDS_TO_SELECT;
    }

    @Override
    protected UserTokenDto parseRow(ResultSet resultSet) {
        try {
            return new UserTokenDto(
                    Option.of(UUID.fromString(resultSet.getString("id"))),
                    Option.ofNullable(resultSet.getString("uid")),
                    UUID.fromString(resultSet.getString("conference_id")),
                    resultSet.getString("token"),
                    new Instant(resultSet.getTimestamp("created_at")),
                    resultSet.getString("status")
            );
        } catch (SQLException e) {
            throw new TelemostRuntimeException(e);
        }
    }

    @Override
    public UserTokenDto insert(UserTokenDto userTokenDto) {
        MapF<String, ?> params = Cf.map(
                "conference_id", userTokenDto.getConferenceId(),
                "token", userTokenDto.getToken(),
                "created_at", userTokenDto.getCreatedAt())
                .plus1("status", userTokenDto.getStatus());
        return getJdbcTemplate().query(queries.getTs(INSERT_ONE), (rs, rowNum) -> parseRow(rs), params).first();
    }

    @Override
    public Option<UserTokenDto> getActiveOrUpdateCreatedToken(UUID conferenceId, String uid, String token, long expireSeconds) {
        MapF<String, ?> params = Cf.map(
                "conference_id", conferenceId,
                "uid", uid,
                "token", token,
                "expiration_seconds", expireSeconds);
        return getJdbcTemplate().queryForOption(queries.getTs(ACTIVATE_TOKEN), (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public Option<UserTokenDto> expireToken(UUID conferenceId, String token, long expirationSecond) {
        MapF<String, ?> params = Cf.map(
                "conference_id", conferenceId,
                "expiration_seconds", expirationSecond,
                "token", token);
        return getJdbcTemplate().queryForOption(queries.getTs(EXPIRE_TOKEN), (rs, rowNum) -> parseRow(rs), params);
    }

    @Override
    public Option<UserTokenDto> getToken(UUID conferenceId, String uid, String token, long expireSeconds) {
        MapF<String, ?> params = Cf.map(
                "conference_id", conferenceId,
                "uid", uid,
                "token", token,
                "expiration_seconds", expireSeconds);
        return getJdbcTemplate().queryForOption(queries.getTs(GET_TOKEN), (rs, rowNum) -> parseRow(rs), params);
    }
}
