package ru.yandex.chemodan.app.telemost.room;

import java.util.concurrent.CompletableFuture;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.logging.TelemostAuditLog;
import ru.yandex.chemodan.app.telemost.room.proto.MediatorOuterClass;
import ru.yandex.chemodan.app.telemost.services.model.Conference;
import ru.yandex.chemodan.app.telemost.services.model.User;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public abstract class AbstractRoomManager implements RoomManager {

    private static final Logger logger = LoggerFactory.getLogger(AbstractRoomManager.class);

    public static final String VIDEO_CONSTRAINTS_MAIL_PRO_EXPERIMENT_FLAG = "disk-mailpro";

    private final ExperimentsManager experimentsManager;

    private final TelemostAuditLog telemostAuditLog;

    public AbstractRoomManager(ExperimentsManager experimentsManager, TelemostAuditLog telemostAuditLog) {
        this.experimentsManager = experimentsManager;
        this.telemostAuditLog = telemostAuditLog;
    }

    protected MediatorOuterClass.AddParticipantRequest createAddParticipantRequest(String peerId, String roomId,
            boolean forceJoin, Option<User> user)
    {
        MediatorOuterClass.AddParticipantRequest.Builder requestBuilder =
                MediatorOuterClass.AddParticipantRequest.newBuilder()
                        .setParticipantId(peerId)
                        .setForceJoin(forceJoin)
                        .setRoomId(roomId);
        Option<MediatorOuterClass.VideoConstraints> videoConstraints = getVideoConstraintsForUser(user);
        if (videoConstraints.isPresent()) {
            requestBuilder = requestBuilder.setVideoConstraints(videoConstraints.get());
        }
        return requestBuilder.build();
    }

    private Option<MediatorOuterClass.VideoConstraints> getVideoConstraintsForUser(Option<User> userO) {
        if (!userO.isPresent()) {
            return Option.empty();
        }
        User user = userO.get();
        if (user.isMailPro()) {
            return Option.of(createFullHdVideoConstraints());
        }
        if (!user.getUid().isPassportUid()) {
            return Option.empty();
        }
        if (isUserConnectedToExperiment(user.getUid().getPassportUid())) {
            return Option.of(createFullHdVideoConstraints());
        }
        return Option.empty();
    }

    private MediatorOuterClass.VideoConstraints createFullHdVideoConstraints() {
        return MediatorOuterClass.VideoConstraints.newBuilder().setHighDefinitionP2P(true).build();
    }

    private boolean isUserConnectedToExperiment(PassportUid uid) {
        try {
            return experimentsManager.getFlags(uid.toUidOrZero().getUid())
                    .containsTs(VIDEO_CONSTRAINTS_MAIL_PRO_EXPERIMENT_FLAG);
        } catch (Exception e) {
            logger.info("Cannot fetch data from UaaS", e);
            return false;
        }
    }

    protected abstract void createRoomImpl(Conference conference);

    @Override
    public void createRoom(Conference conference) {
        logger.info("call createRoom: conference={}", conference.getRoomId());
        createRoomImpl(conference);
        telemostAuditLog.logConferenceCreated(conference);
    }

    protected CompletableFuture<?> failedFuture(Throwable t){
        CompletableFuture<?> cf = new CompletableFuture();
        cf.completeExceptionally(t);
        return cf;
    }
}
