package ru.yandex.chemodan.app.telemost.room;

import io.grpc.Status;
import io.grpc.StatusRuntimeException;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.logging.TelemostAuditLog;
import ru.yandex.chemodan.app.telemost.room.proto.MediatorOuterClass;
import ru.yandex.chemodan.app.telemost.room.proto.RoomGrpc;
import ru.yandex.chemodan.app.telemost.services.model.Conference;
import ru.yandex.chemodan.app.telemost.services.model.User;
import ru.yandex.chemodan.app.uaas.experiments.ExperimentsManager;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class RoomBlockingManager extends AbstractRoomManager {

    private static final Logger logger = LoggerFactory.getLogger(RoomBlockingManager.class);

    private final RoomGrpc.RoomBlockingStub roomBlockingStub;

    private final RoomSettingsHolder roomSettingsHolder;

    public RoomBlockingManager(RoomGrpc.RoomBlockingStub roomBlockingStub, RoomSettingsHolder roomSettingsHolder,
                               ExperimentsManager experimentsManager, TelemostAuditLog telemostAuditLog) {
        super(experimentsManager, telemostAuditLog);
        this.roomBlockingStub = roomBlockingStub;
        this.roomSettingsHolder = roomSettingsHolder;
    }

    @Override
    public void createRoomImpl(Conference conference) {
        try {
            roomBlockingStub.createRoom(MediatorOuterClass.CreateRoomRequest.newBuilder()
                    .setAppServerEndpoint(roomSettingsHolder.getAppServerEndpoint())
                    .setRoomId(conference.getRoomId())
                    .build());
        } catch (StatusRuntimeException e) {
            if (!Status.ALREADY_EXISTS.getCode().equals(e.getStatus().getCode())) {
                throw new RuntimeException(e);
            }
        }
    }

    @Override
    public void addParticipant(Conference conference, String peerId, boolean forceJoin,
                                               Option<User> user) {
        final String roomId = conference.getRoomId();
        try {
            roomBlockingStub.addParticipant(createAddParticipantRequest(peerId, roomId, forceJoin, user));
        } catch (StatusRuntimeException e) {
            if (Status.Code.ALREADY_EXISTS.equals(e.getStatus().getCode())) {
                logger.warn("User {} already exists in mediator, skipping adding error", peerId);
            } else if (Status.Code.NOT_FOUND.equals(e.getStatus().getCode())) {
                logger.info("Room doesn't exist, creating room {}", roomId);
                createRoom(conference);
                addParticipant(conference, peerId, forceJoin, user);
            } else {
                throw new RuntimeException(e);
            }
        }
    }

    @Override
    public void removeParticipant(Conference conference, String peerId) {
        roomBlockingStub.removeParticipant(MediatorOuterClass.RemoveParticipantRequest.newBuilder()
                .setRoomId(conference.getRoomId()).setParticipantId(peerId).build());
    }
}
