package ru.yandex.chemodan.app.telemost.services;

import java.util.UUID;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.exceptions.NoSuchBroadcastCreatedException;
import ru.yandex.chemodan.app.telemost.exceptions.StreamNotStartedException;
import ru.yandex.chemodan.app.telemost.exceptions.TelemostRuntimeException;
import ru.yandex.chemodan.app.telemost.repository.dao.BroadcastDao;
import ru.yandex.chemodan.app.telemost.repository.dao.StreamDao;
import ru.yandex.chemodan.app.telemost.repository.model.BroadcastDto;
import ru.yandex.chemodan.app.telemost.repository.model.StreamDto;
import ru.yandex.chemodan.app.telemost.services.model.Broadcast;
import ru.yandex.chemodan.app.telemost.web.v2.model.BroadcastInitData;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class BroadcastService {
    private static final Logger logger = LoggerFactory.getLogger(BroadcastService.class);

    private final int insertBroadcastRetryCount;

    private final BroadcastUriService broadcastUriService;
    private final BroadcastDao broadcastDao;
    private final StreamDao streamDao;

    public Broadcast createBroadcast(UUID conferenceId, String uid, BroadcastInitData broadcastInitData) {
        BroadcastDto broadcastDto = getByConferenceId(conferenceId).map(t -> t).orElse(
                upsertBroadcast(conferenceId, uid, generateBroadcastKey(), Option.empty(),
                        broadcastInitData.getCaption(), broadcastInitData.getDescription())
        );

        String uri = broadcastUriService.buildBroadcastUrl(broadcastDto.getBroadcastKey());
        return new Broadcast(broadcastDto, uri, Option.empty());
    }

    private String generateBroadcastKey() {
        return UUID.randomUUID().toString().replace("-", "");
    }

    private BroadcastDto upsertBroadcast(UUID conferenceId, String createdBy, String broadcastKey,
                                         Option<UUID> broadcastChatId, Option<String> caption,
                                         Option<String> description) {
        BroadcastDto broadcastDto = new BroadcastDto(
                conferenceId,
                createdBy,
                broadcastKey,
                broadcastChatId,
                caption,
                description
        );
        logger.info("Upsert broadcast for conferenceId={}: {}", conferenceId, broadcastDto);
        return RetryUtils.retry(logger, insertBroadcastRetryCount, () ->  broadcastDao.upsert(broadcastDto));
    }

    public Option<BroadcastDto> getById(UUID broadcastId) {
        return broadcastDao.findByBroadcastId(broadcastId);
    }

    public Option<BroadcastDto> getByKey(String broadcastKey) {
        return broadcastDao.findByBroadcastKey(broadcastKey);
    }

    public Option<BroadcastDto> getByConferenceId(UUID conferenceId) {
        return broadcastDao.findByConferenceId(conferenceId);
    }

    public void setUgcLiveLine(UUID broadcastId, long ugcLiveLine) {
        if (!broadcastDao.updateUgcLiveLineIfNull(broadcastId, ugcLiveLine)) {
            throw new TelemostRuntimeException("Already have line or no such broadcast " + broadcastId);
        }
    }

    public void setBroadcastChatId(UUID broadcastId, UUID broadcastChatId) {
        if (!broadcastDao.updateBroadcastChatIdIfNull(broadcastId, broadcastChatId)) {
            throw new TelemostRuntimeException("Already have chat or no such broadcast " + broadcastId);
        }
    }

    public boolean checkTranslatorToken(UUID conferenceId, String token) {
        StreamDto streamDto = getActiveStream(conferenceId);
        if (streamDto.getTranslatorToken().isEmpty()) {
            return true;
        }
        return streamDto.getTranslatorToken().get().equals(token);
    }

    public void updateTranslatorPeerIfNeed(UUID conferenceId, String peerId) {
        StreamDto streamDto = getActiveStream(conferenceId);
        if (streamDto.getTranslatorPeerId().isEmpty() || !streamDto.getTranslatorPeerId().get().equals(peerId)) {
            RetryUtils.retry(logger, insertBroadcastRetryCount,
                    () -> streamDao.update(new StreamDto(streamDto, Option.of(peerId))));
        }
    }

    private StreamDto getActiveStream(UUID conferenceId) {
        return streamDao.findActiveByBroadcastKey(broadcastDao.findByConferenceId(conferenceId)
                        .orElseThrow(NoSuchBroadcastCreatedException::new).getBroadcastKey())
                .orElseThrow(StreamNotStartedException::new);
    }
}
