package ru.yandex.chemodan.app.telemost.services;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.calendar.CalendarClient;
import ru.yandex.chemodan.app.telemost.calendar.model.CalendarEvent;
import ru.yandex.chemodan.app.telemost.exceptions.ConferenceNotFoundTelemostException;
import ru.yandex.chemodan.app.telemost.exceptions.ForbiddenAccessToPrivateConferenceException;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceUserDao;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceDto;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceUserDto;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class CalendarService {
    private static final Logger logger = LoggerFactory.getLogger(CalendarService.class);

    private final CalendarClient calendarClient;

    private final ConferenceUserDao conferenceUserDao;

    public CalendarService(CalendarClient calendarClient, ConferenceUserDao conferenceUserDao) {
        this.calendarClient = calendarClient;
        this.conferenceUserDao = conferenceUserDao;
    }

    public ConferenceDto getCalendarEventData(ConferenceDto conference, Option<PassportOrYaTeamUid> uid,
                                              Option<String> tvmUserTicket, boolean failOnError) {
        if (!conference.getEventId().isPresent()) {
            return conference;
        }

        // TODO: До доработок бэка публичного календаря uid и user-ticket не используем. Получение
        //       события выполняется от имени владельца встречи без user-ticket.
        Option<ConferenceUserDto> owner = conferenceUserDao.findOwner(conference.getId());
        if (owner.isPresent()) {
            PassportOrYaTeamUid ownerUid = PassportOrYaTeamUid.parseUid(owner.get().getUid());
            uid = Option.of(ownerUid);
            tvmUserTicket = Option.empty();
        }

        RuntimeException toThrow = null;
        try {
            CalendarEvent event = calendarClient.getEvent(conference.getEventId().get(), uid, tvmUserTicket);
            logger.info("Got calendar event for conference id={}, event id={}: event data={}",
                    conference.getConferenceId(), conference.getEventId(), event);
            return conference.withCalendarEventData(event);
        } catch (ConferenceNotFoundTelemostException e) {
            logger.info("Calendar event was deleted (or not found) for conference id={}, event id={}: {}",
                    conference.getConferenceId(), conference.getEventId(), e.toString());
            toThrow = e;
        } catch (Exception e) {
            logger.error("Error on got calendar event for conference id={}, event id={}",
                    conference.getConferenceId(), conference.getEventId(), e);
            toThrow = new ForbiddenAccessToPrivateConferenceException();
        }
        if (!failOnError) {
            return conference;
        }
        throw toThrow;
    }
}
