package ru.yandex.chemodan.app.telemost.services;

import java.util.UUID;
import java.util.concurrent.CompletableFuture;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.format.DateTimeFormat;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.telemost.appmessages.AppMessageSender;
import ru.yandex.chemodan.app.telemost.appmessages.model.ChatCreated;
import ru.yandex.chemodan.app.telemost.chat.ChatClient;
import ru.yandex.chemodan.app.telemost.chat.model.Chat;
import ru.yandex.chemodan.app.telemost.chat.model.ChatHistory;
import ru.yandex.chemodan.app.telemost.chat.model.ChatRole;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceDtoDao;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceStateDao;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceUserDao;
import ru.yandex.chemodan.app.telemost.repository.model.BroadcastDto;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceDto;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceStateDto;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceUserDto;
import ru.yandex.chemodan.app.telemost.repository.model.StreamDto;
import ru.yandex.chemodan.app.telemost.repository.model.UserRole;
import ru.yandex.chemodan.app.telemost.services.model.ChatType;
import ru.yandex.chemodan.app.telemost.services.model.Conference;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.chemodan.app.telemost.services.model.User;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.tanker.TankerClient;
import ru.yandex.inside.tanker.model.TankerKey;
import ru.yandex.inside.tanker.model.TankerResponse;
import ru.yandex.inside.tanker.model.TankerTranslation;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.MoscowTime;

public class ChatService {
    public static final long DEFAULT_BROADCAST_CHAT_TBL_SECS = 3600;

    private static final Logger logger = LoggerFactory.getLogger(ChatService.class);

    private static final String SYSTEM_MESSAGE_KEY = "system_message";
    private static final String PERMANENT_TITLE_KEY = "permanent_title";
    private static final String PERMANENT_DESCRIPTION_KEY = "permanent_description";
    private static final String NONPERMANENT_TITLE_KEY = "nonpermanent_title";
    private static final String NONPERMANENT_DESCRIPTION_KEY = "nonpermanent_description";

    private static final String BROADCAST_TITLE_KEY = "broadcast_title";
    private static final String BROADCAST_DESCRIPTION_KEY = "broadcast_description";
    private static final String NONAME_TITLE_KEY = "noname_title";
    private static final String BROADCAST_SYSTEM_MESSAGE_KEY = "broadcast_system_message";

    private static final int CHAT_TEXT_LENGTH_LIMIT = 256;

    // Используется на случай недоступности Танкера
    private static final String DEFAULT_TITLE = "Telemost %date%";

    private final ChatClient chatClient;
    private final ConferenceDtoDao conferenceDtoDao;
    private final ConferenceUserDao conferenceUserDao;
    private final ConferenceStateDao conferenceStateDao;
    private final ConferenceService conferenceService;
    private final ConferenceUriService conferenceUriService;
    private final ConferencePeerService conferencePeerService;
    private final CalendarService calendarService;
    private final BroadcastService broadcastService;
    private final BroadcastUriService broadcastUriService;
    private final StreamService streamService;
    private final UUID bot;
    private final TankerClient tankerClient;

    private final ListF<String> chatAvatarIds;
    private final AppMessageSender appMessageSender;

    private final DynamicProperty<Long> broadcastChatTbl =
            new DynamicProperty<>("telemost-broadcast-chat-tbl-secs", DEFAULT_BROADCAST_CHAT_TBL_SECS);

    public ChatService(ConferenceService conferenceService, ConferenceUriService conferenceUriService, ChatClient chatClient,
                       ConferenceDtoDao conferenceDtoDao, ConferenceUserDao conferenceUserDao,
                       ConferenceStateDao conferenceStateDao, AppMessageSender appMessageSender,
                       ConferencePeerService conferencePeerService, CalendarService calendarService,
                       BroadcastService broadcastService, BroadcastUriService broadcastUriService, StreamService streamService,
                       ListF<String> chatAvatarIds, UUID bot, TankerClient tankerClient) {
        this.conferenceService = conferenceService;
        this.conferenceUriService = conferenceUriService;
        this.chatClient = chatClient;
        this.conferenceDtoDao = conferenceDtoDao;
        this.conferenceUserDao = conferenceUserDao;
        this.conferenceStateDao = conferenceStateDao;
        this.appMessageSender = appMessageSender;
        this.conferencePeerService = conferencePeerService;
        this.calendarService = calendarService;
        this.broadcastService = broadcastService;
        this.broadcastUriService = broadcastUriService;
        this.streamService = streamService;
        this.chatAvatarIds = chatAvatarIds;
        this.bot = bot;
        this.tankerClient = tankerClient;
    }

    public Option<UUID> getUser(String uid) {
        return chatClient.getUser(uid);
    }

    private UUID getOrGenerateChatId(ConferenceDto conferenceDto, Option<BroadcastDto> broadcastDtoO, ChatType chatType) {
        if (chatType == ChatType.CONFERENCE) {
            return conferenceDto.getId();
        }
        return broadcastDtoO.get().getBroadcastChatId().map(x -> x).orElse(UUID.randomUUID());
    }

    private UUID getChatId(ConferenceDto conferenceDto, Option<BroadcastDto> broadcastDtoO, ChatType chatType) {
        if (chatType == ChatType.CONFERENCE) {
            return conferenceDto.getId();
        }
        return broadcastDtoO.get().getBroadcastChatId().get();
    }

    private Option<Chat> checkExistingChat(ConferenceDto conferenceDto, UUID chatId, ChatType chatType, UUID user,
                                           Option<PassportOrYaTeamUid> uid) {
        Option<Chat> existingChatO = chatClient.getChats(chatId, chatType);
        if (existingChatO.isPresent()) {

            logger.info("Create chat: chat found for conference={}, type={}, uid={}, user={}, chat_path={}",
                    conferenceDto.getConferenceId(), chatType, uid, user, existingChatO.get().getChatPath());

            if (!existingChatO.get().getMembers().containsTs(user)) {
                ChatRole chatRole = getChatRole(uid, conferenceDto);
                existingChatO = chatClient.updateMembers(chatId, chatType, Cf.list(user), Cf.list(), chatRole);
            }

            // Add into conference state and send app-message
            postHandleChatCreation(existingChatO.get(), conferenceDto);
            return existingChatO;
        }
        return Option.empty();
    }

    private String getTextByKey(MapF<String, TankerKey> tankerKeys, String key, String lang) {
        String text = "";
        MapF<String, TankerTranslation> translations = tankerKeys.getO(key).get().translations;
        if (translations.getO(lang).isPresent()) {
            text = translations.getO(lang).get().form.get().text;
        }
        if (text.isEmpty()) {
            text = translations.getO("ru").get().form.get().text;
        }
        return text;
    }

    private String getTitle(ConferenceDto conferenceDto, Option<BroadcastDto> broadcastDtoO, ChatType chatType,
                            MapF<String, TankerKey> tankerKeys, DateTimeZone tz, String lang) {
        // 1. Если конференция или трансляция из Календаря, то заголовок - это заголовок из события Календаря
        if (conferenceDto.getEventCaption().isPresent()) {
            return replaceTemplateParams(getChatTextBasedOnLengthLimit(conferenceDto.getEventCaption().get()),
                    conferenceDto, broadcastDtoO, tz);
        }

        String title = DEFAULT_TITLE;

        if (chatType == ChatType.CONFERENCE && tankerKeys.isNotEmpty()) {
            // 2. Если это чат конференции, то берем по заданному алгоритму ключи из Танкера
            title = getTextByKey(tankerKeys,
                    conferenceDto.isPermanent() ? PERMANENT_TITLE_KEY : NONPERMANENT_TITLE_KEY, lang);
        } else if (chatType == ChatType.BROADCAST) {
            // 3. Если это чат трансляции, то берем по заданному алгоритму заголовок:
            //    - заголовок, если пользователь его ввел
            //    - "Без названия", если пользователь ввел описание
            //    - текст из Танкера в ином случае
            //    - если же даже в Танкере ключей нет (не удалось ранее получить), то дефолтное значение
            // При этом трансляция должна существовать до начала создания трансляционного чата
            title = broadcastDtoO.get().getCaption().map(x -> x).orElse("");
            if (title.isEmpty()) {
                if (tankerKeys.isNotEmpty()) {
                    title = getTextByKey(tankerKeys,
                            broadcastDtoO.get().getDescription().isEmpty() ? BROADCAST_TITLE_KEY : NONAME_TITLE_KEY, lang);
                } else {
                    title = DEFAULT_TITLE;
                }
            }
        }

        return replaceTemplateParams(title, conferenceDto, broadcastDtoO, tz);
    }

    private String getDescription(ConferenceDto conferenceDto, Option<BroadcastDto> broadcastDtoO, ChatType chatType,
                                  MapF<String, TankerKey> tankerKeys, DateTimeZone tz, String lang) {
        // 1. Если это чат конференции, которая создана из календаря, то описание чата это описание события из календаря
        if (chatType == ChatType.CONFERENCE && conferenceDto.getEventDescription().isPresent()) {
            return replaceTemplateParams(getChatTextBasedOnLengthLimit(conferenceDto.getEventDescription().get()),
                    conferenceDto, broadcastDtoO, tz);
        }

        String description = "";

        if (chatType == ChatType.CONFERENCE && tankerKeys.isNotEmpty()) {
            // 2. Если это чат конференции, то описание берем из Танкера по заданному алгоритму
            description = getTextByKey(tankerKeys,
                    conferenceDto.isPermanent() ? PERMANENT_DESCRIPTION_KEY : NONPERMANENT_DESCRIPTION_KEY, lang);
        } else if (chatType == ChatType.BROADCAST && conferenceDto.getEventId().isEmpty()) {
            // 3. Если это чат трансляции, то описание формируем по следующему алгоритму:
            //    - если трансляция создана из Календаря, то описание пустое
            //    - если для трансляции указано название и/или описание, то берем указанное описание (может быть пустым)
            //    - если название и описание пустые, то берем описание из Танкера
            if (broadcastDtoO.get().getCaption().isPresent() || broadcastDtoO.get().getDescription().isPresent()) {
                description = broadcastDtoO.get().getDescription().map(x -> x).orElse("");
            } else if (tankerKeys.isNotEmpty()) {
                description = getTextByKey(tankerKeys, BROADCAST_DESCRIPTION_KEY, lang);
            }
        }
        return replaceTemplateParams(description, conferenceDto, broadcastDtoO, tz);
    }

    public Option<Chat> tryCreateBroadcastChatIfNeed(String broadcastUri) {
        logger.info("Create broadcast chat for broadcast uri={}", broadcastUri);
        Option<BroadcastDto> broadcastDtoO = broadcastService.getByKey(
                broadcastUriService.getBroadcastUriData(broadcastUri).getBroadcastKey());
        if (broadcastDtoO.isEmpty() || broadcastDtoO.get().getBroadcastChatId().isPresent()) {
            return Option.empty();
        }

        ConferenceDto conferenceDto = conferenceService.findById(broadcastDtoO.get().getConferenceId());
        try {
            return checkAndCreateBroadcastChat(conferenceDto, Option.empty(), Option.empty());
        } catch (Exception e) {
            logger.error("Error: Failed to create chat for broadcast {}, conference {}", broadcastDtoO, conferenceDto, e);
            return Option.empty();
        }
    }

    public Option<Chat> tryCreateBroadcastChatIfNeed(ConferenceDto conferenceDto, Option<PassportOrYaTeamUid> uid,
                                                     Option<String> tvmUserTicket) {
        logger.info("Create broadcast chat for conference id={}, uid={}", conferenceDto.getConferenceId(),
                uid);
        Option<BroadcastDto> broadcastDtoO = broadcastService.getByConferenceId(conferenceDto.getId());
        if (broadcastDtoO.isEmpty() || broadcastDtoO.get().getBroadcastChatId().isPresent()) {
            return Option.empty();
        }

        try {
            return checkAndCreateBroadcastChat(conferenceDto, uid, tvmUserTicket);
        } catch (Exception e) {
            logger.error("Error: Failed to create chat for UID {}, broadcast {}, conference {}", uid, broadcastDtoO, conferenceDto, e);
            return Option.empty();
        }
    }

    private Option<Chat> checkAndCreateBroadcastChat(ConferenceDto conferenceDto, Option<PassportOrYaTeamUid> uidO,
                                             Option<String> tvmUserTicket) {
        // Условия для создания чата трансляции:
        // 1. Трансляция для данной конференции создана (проверяется выше)
        // 2. В трансляции еще отсутствует трансляционный чат (проверяется выше)
        // 3. Трансляция НЕ подвязана к событию календаря или до начала события осталось менее часа
        // 4. При этом чаты должны быть разрешены
        boolean isStaffOwner = conferenceService.isStaffOwner(conferenceDto);
        if (!conferenceService.isBroadcastAllowed(conferenceDto, isStaffOwner)
                || !conferenceService.isChatsAllowed(conferenceDto, isStaffOwner)) {
            return Option.empty();
        }
        if (conferenceDto.getEventId().isPresent()) {
            conferenceDto = calendarService.getCalendarEventData(conferenceDto, uidO, tvmUserTicket, false);
            if (conferenceDto.getStartEvent().exists(
                    x -> x.minus(Duration.standardSeconds(broadcastChatTbl.get())).isAfterNow())
            ) {
                return Option.empty();
            }
        }

        Option<String> ownerUidO = conferenceUserDao.findOwner(conferenceDto.getId()).map(ConferenceUserDto::getUid);

        Option<UUID> userO = uidO.map(PassportOrYaTeamUid::asString)
                .orElse(ownerUidO)
                .flatMapO(this::getUser);

        if (userO.isPresent()) {
            return createChat(conferenceDto.getConferenceId(), ChatType.BROADCAST, userO.get(),
                    ownerUidO.map(PassportOrYaTeamUid::parseUid), tvmUserTicket);
        }
        return Option.empty();
    }

    public Option<Chat> createChat(String conferenceId, ChatType chatType, UUID user, Option<PassportOrYaTeamUid> uid,
                                   Option<String> tvmUserTicket) {
        // Get conference data
        Option<ConferenceDto> conferenceDtoO = conferenceDtoDao.findByConferenceId(conferenceId);
        if (!conferenceDtoO.isPresent()) {
            return Option.empty();
        }

        logger.info("Create chat: conference found for conference={}, type={}, uid={}, user={}",
                conferenceDtoO.get().getConferenceId(), chatType, uid, user);

        return createChat(conferenceDtoO.get(), chatType, user, uid, tvmUserTicket);
    }

    private Option<Chat> createChat(ConferenceDto conferenceDto, ChatType chatType, UUID user, Option<PassportOrYaTeamUid> uid,
                                   Option<String> tvmUserTicket) {

        logger.info("Create chat: request for conference={}, type={}, uid={}, user={}",
                conferenceDto.getConferenceId(), chatType, uid, user);

        // Check access to calendar event
        conferenceDto = calendarService.getCalendarEventData(conferenceDto, uid, tvmUserTicket, false);

        // Check broadcast, if this broadcast chat
        Option<BroadcastDto> broadcastDtoO = Option.empty();
        if (chatType == ChatType.BROADCAST) {
            broadcastDtoO = broadcastService.getByConferenceId(conferenceDto.getId());
        }

        // Get or generate chat Id
        UUID chatId = getOrGenerateChatId(conferenceDto, broadcastDtoO, chatType);

        // Check existing chat
        Option<Chat> chatO = checkExistingChat(conferenceDto, chatId, chatType, user, uid);
        if (chatO.isPresent()) {
            return chatO;
        }

        // Get conference admins, members and subscribers
        //
        // Логика для админов:
        // 1. Добавляем в чат всех админов
        //
        // Логика для members/subscribers:
        // 1. Если админ есть, то он должен быть и в members (не считая бота)
        // 2. Если uid присутствует, значит это авторизованный пользователь - должен быть в members
        // 3. Если uid нет, значит это подписчик - должен быть в subscribers
        ListF<UUID> admins = conferenceUserDao.findByConference(conferenceDto.getId()).stream()
                .filter(conferenceUser -> conferenceUser.getRole().getChatRole().equals(ChatRole.ADMIN))
                .map(ConferenceUserDto::getUid)
                .map(chatClient::getUser)
                .filter(Option::isPresent)
                .map(Option::get)
                .collect(CollectorsF.toArrayList());
        ListF<UUID> members = Cf.toList(admins);
        admins.add(bot);

        // Find user
        DateTimeZone tz = MoscowTime.TZ;
        String lang = "ru";
        if (uid.isPresent()) {
            try {
                User userData = conferencePeerService.findUser(uid.get());
                tz = userData.getTimezone();
                lang = userData.getLanguage();
            } catch (Exception ignored) { }
        }

        // Get tanker keys
        MapF<String, TankerKey> tankerKeys = Cf.hashMap();
        try {
            TankerResponse tankerResponse = tankerClient.retrieveDataByKeysetId("telemost", "chats", null);
            tankerKeys = tankerResponse.keysets.getO("chats").get().keys;
        } catch (Exception e) {
            logger.info("Create chat: error receiving data from Tanker for conference={}: {}. Using default values.",
                    conferenceDto.getConferenceId(), e);
        }

        // Formatting title
        String title = getTitle(conferenceDto, broadcastDtoO, chatType, tankerKeys, tz, lang);

        // Formatting description
        String description = getDescription(conferenceDto, broadcastDtoO, chatType, tankerKeys, tz, lang);

        // Random selection avatar
        Option<String> avatarId = Option.empty();
        if (chatAvatarIds.size() > 0) {
            int idx = (int)(DateTime.now().toInstant().getMillis() % 1000);
            avatarId = Option.of(chatAvatarIds.get(idx % chatAvatarIds.size()));
        }

        // Create chat
        logger.info("Create chat: new chat for conference={}, type={}, uid={}, user={}, admins={}, members={}",
                conferenceDto.getConferenceId(), chatType, uid, user, admins, members);

        chatClient.createChat(chatId, chatType, title, description, avatarId, admins, members);
        if (!admins.containsTs(user) && !members.containsTs(user)) {
            chatClient.updateMembers(chatId, chatType, Cf.list(user), Cf.list(), getChatRole(uid, conferenceDto));
        }

        // Push system message
        if (tankerKeys.isNotEmpty()) {
            String systemMessage = getTextByKey(tankerKeys,
                    chatType == ChatType.CONFERENCE ? SYSTEM_MESSAGE_KEY : BROADCAST_SYSTEM_MESSAGE_KEY, lang);
            chatClient.push(chatId, chatType, bot, replaceTemplateParams(systemMessage, conferenceDto, broadcastDtoO,
                    tz));
        }

        Option<Chat> createdChatO = chatClient.updateMembers(chatId, chatType, Cf.list(), Cf.list(bot), ChatRole.ADMIN);

        // Store chat-id
        if (chatType == ChatType.BROADCAST && broadcastDtoO.get().getBroadcastChatId().isEmpty()) {
            broadcastService.setBroadcastChatId(broadcastDtoO.get().getId().get(), chatId);
        }

        logger.info("Create chat: system message pushed for conference={}, type={}, uid={}, user={}",
                conferenceDto.getConferenceId(), chatType, uid, user);

        // Add into conference state and send message
        if (createdChatO.isPresent()) {
            postHandleChatCreation(createdChatO.get(), conferenceDto);
        }

        return createdChatO;
    }

    private String replaceTemplateParams(String template, ConferenceDto conferenceDto,
                                         Option<BroadcastDto> broadcastDtoO, DateTimeZone tz) {
        template = template.replace("%conference-uri%", conferenceUriService.buildConferenceUrl(conferenceDto));
        if (broadcastDtoO.isPresent()) {
            template = template.replace("%broadcast-uri%", broadcastDtoO.get().getBroadcastUri());
        }
        template = template.replace("%date%", conferenceDto.isPermanent() ?
                    DateTime.now().toString(DateTimeFormat.forPattern("dd.MM в HH:mm").withZone(tz)) :
                    conferenceDto.getCreatedAt().toString(DateTimeFormat.forPattern("dd.MM в HH:mm").withZone(tz)));
        return template;
    }

    private String getChatTextBasedOnLengthLimit(String t) {
        if (t.length() > CHAT_TEXT_LENGTH_LIMIT) {
            return t.substring(0, CHAT_TEXT_LENGTH_LIMIT - 3) + "...";
        }
        return t;
    }

    private CompletableFuture<?> postHandleChatCreation(Chat chat, ConferenceDto conferenceDto) {
        updateConferenceStateWithChat(chat, conferenceDto);
        return sendNotification(chat, conferenceDto);
    }

    private void updateConferenceStateWithChat(Chat chat, ConferenceDto conference) {
        boolean isStaffOwner = conferenceService.isStaffOwner(conference);

        Option<ConferenceStateDto> conferenceStateO = conferenceStateDao.findState(conference.getId());
        Option<BroadcastDto> broadcastDtoO = broadcastService.getByConferenceId(conference.getId());
        Option<StreamDto> streamDtoO = Option.empty();
        if (broadcastDtoO.isPresent()) {
            streamDtoO = streamService.findActiveStreamByBroadcastKey(broadcastDtoO.get().getBroadcastKey());
        }

        ConferenceStateDto conferenceState = conferenceStateO.isPresent() ? conferenceStateO.get() :
                new ConferenceStateDto(conferenceService.isLocalRecordingAllowed(),
                        conferenceService.isCloudRecordingAllowed(),
                        conferenceService.isChatsAllowed(conference, isStaffOwner),
                        Option.of(chat.getChatPath()),
                        conferenceService.isControlAllowed(conference, isStaffOwner),
                        conferenceService.isBroadcastAllowed(conference, isStaffOwner),
                        conferenceService.isBroadcastFeatureEnabled(conference),
                        broadcastDtoO,
                        streamDtoO,
                        conference.getId()
                );

        if (chat.getChatType() == ChatType.CONFERENCE) {
            conferenceState.setChatPath(Option.of(chat.getChatPath()));
            conferenceStateDao.updateVersion(conferenceState);
        } else {
            conferenceStateDao.incrementVersion(conference.getId());
        }

        logger.info("Update conference state: chat={}, conference={}, state={}",
                chat.getChatPath(), conference.getConferenceId(), conferenceState);
    }

    private CompletableFuture<?> sendNotification(Chat chat, ConferenceDto conferenceDto) {
        CompletableFuture<?> future = appMessageSender.sendMessageToAllAsync(
                conferenceDto.getConferenceId(), new ChatCreated(chat.getChatType(), chat.getChatPath()));
        logger.info("Send notify message about chat created for conference={}, chat_type={}, chat_path={}",
                conferenceDto.getConferenceId(), chat.getChatType(), chat.getChatPath());
        return future;
    }

    public Option<Chat> joinToChatIfExisting(String conferenceId, ChatType chatType, UUID user, Option<PassportOrYaTeamUid> uid,
                                             Option<String> tvmUserTicket) {

        logger.info("Join to chat: request for conference={}, type={}, uid={}, user={}", conferenceId,
                chatType, uid, user);

        // Get conference data
        Option<ConferenceDto> conferenceDtoO = conferenceDtoDao.findByConferenceId(conferenceId);
        if (!conferenceDtoO.isPresent()) {
            return Option.empty();
        }

        logger.info("Join to chat: conference found for conference={}, type={}, uid={}, user={}", conferenceId,
                chatType, uid, user);

        // Check access to calendar event
        ConferenceDto conferenceDto = conferenceDtoO.get();
        conferenceDto = conferenceService.conferenceAcceptableByCalendarEventParameter(
                    conferenceDto, uid, tvmUserTicket, false
        );

        // Check broadcast, if this broadcast chat
        Option<BroadcastDto> broadcastDtoO = Option.empty();
        if (chatType == ChatType.BROADCAST) {
            broadcastDtoO = broadcastService.getByConferenceId(conferenceDto.getId());
            if (broadcastDtoO.isEmpty()) {
                return Option.empty();
            }
        }
        UUID chatId = getChatId(conferenceDto, broadcastDtoO, chatType);

        // Check chat existing
        Option<Chat> existingChatO = chatClient.getChats(chatId, chatType);
        if (!existingChatO.isPresent()) {
            return Option.empty();
        }
        if (!existingChatO.get().getMembers().containsTs(user)) {
            ChatRole chatRole = getChatRole(uid, conferenceDto);
            existingChatO = chatClient.updateMembers(chatId, chatType, Cf.list(user), Cf.list(), chatRole);
        }

        logger.info("User uid={}, user={} joined to existing chat_path={}",
                uid, user, existingChatO.get().getChatPath());

        return existingChatO;
    }

    private ChatRole getChatRole(Option<PassportOrYaTeamUid> uid, ConferenceDto conferenceDto) {
        ChatRole chatRole;
        if (uid.isPresent()) {
            chatRole = conferenceUserDao.findByConferenceAndUid(conferenceDto.getId(), uid.get())
                    .map(Function.identityF())
                    .map(ConferenceUserDto::getRole)
                    .map(UserRole::getChatRole)
                    .orElse(ChatRole.MEMBER);
        } else {
            chatRole = ChatRole.SUBSCRIBER;
        }
        return chatRole;
    }

    public Option<ChatHistory> getChatHistory(String conferenceId, ChatType chatType, UUID user, Option<PassportOrYaTeamUid> uid,
                                              Option<Long> offset, Option<String> tvmUserTicket) {
        // Get conference data
        Option<ConferenceDto> conferenceDtoO = conferenceDtoDao.findByConferenceId(conferenceId);
        if (!conferenceDtoO.isPresent()) {
            logger.info("Conference={} for get chat history not found.", conferenceId);
            return Option.empty();
        }

        // Check access to calendar event
        ConferenceDto conferenceDto;
        try {
            conferenceDto = conferenceService.conferenceAcceptableByCalendarEventParameter(
                    conferenceDtoO.get(), uid, tvmUserTicket, true);
        } catch (Exception e) {
            logger.warn("Failed to retrieve calendar event for conference. chats history will not be available. conference={}, uid={}, user={}", conferenceId, uid, user, e);
            return Option.empty();
        }

        // For permanent and old conferences is forbidden to export the history
        if (!conferenceDto.getEventId().isPresent()) {
            Instant limitTime = conferenceDto.getCreatedAt().plus(24 * 3600 * 1000);
            if (conferenceDto.isPermanent() || limitTime.isBeforeNow()) {
                logger.info("Get chat history: is empty because it's permanent or old for conference={}, uid={}, user={}, offset={}, limit_time={}",
                        conferenceId, uid, user, offset, limitTime);
                return Option.empty();
            }
        }

        // Check broadcast, if this broadcast chat
        Option<BroadcastDto> broadcastDtoO = Option.empty();
        if (chatType == ChatType.BROADCAST) {
            broadcastDtoO = broadcastService.getByConferenceId(conferenceDto.getId());
        }
        UUID chatId = getChatId(conferenceDto, broadcastDtoO, chatType);

        // If there is no chat, then we do not get the history
        Option<Chat> existingChatO = chatClient.getChats(chatId, chatType);
        if (!existingChatO.isPresent()) {
            logger.info("Get chat history: is empty because chat not found for conference={}, uid={}, user={}, offset={}",
                    conferenceId, uid, user, offset);
            return Option.empty();
        }

        // Check user uid
        Option<String> uidO = chatClient.getUserUid(user);
        if (uidO.isPresent() != uid.isPresent()) {
            logger.info("Get chat history: is empty because uid present does not match for conference={}, uid={}, user={}, offset={}",
                    conferenceId, uid, user, offset);
            return Option.empty();
        }
        if (uidO.isPresent() && !uid.get().asString().equals(uidO.get())) {
            logger.info("Get chat history: is empty because uid value does not match for conference={}, uid={}, user={}, offset={}",
                    conferenceId, uid, user, offset);
            return Option.empty();
        }

        // Get user and history
        logger.info("Got history for conference={}, user={}, uid={}, offset={}.", conferenceId, user, uid, offset);

        return chatClient.history(chatId, chatType, user, offset);
    }

    public void setRole(ConferenceDto conferenceDto, PassportOrYaTeamUid uid, ChatRole chatRole) {

        Option<BroadcastDto> broadcastDtoO = broadcastService.getByConferenceId(conferenceDto.getId());

        chatClient.getUser(uid.asString()).ifPresent(user -> {
            switch (chatRole) {
                case ADMIN:
                    chatClient.updateMembers(conferenceDto.getId(), ChatType.CONFERENCE, Cf.list(user), Cf.list(),
                            ChatRole.ADMIN);
                    if (broadcastDtoO.isPresent()) {
                        broadcastDtoO.get().getBroadcastChatId().ifPresent(x -> chatClient.updateMembers(
                                x, ChatType.BROADCAST, Cf.list(user), Cf.list(), ChatRole.ADMIN));
                    }
                    break;
                case MEMBER:
                    chatClient.updateMembers(conferenceDto.getId(), ChatType.CONFERENCE, Cf.list(), Cf.list(user),
                            ChatRole.ADMIN);
                    if (broadcastDtoO.isPresent()) {
                        broadcastDtoO.get().getBroadcastChatId().ifPresent(x -> chatClient.updateMembers(
                                x, ChatType.CONFERENCE, Cf.list(), Cf.list(user), ChatRole.ADMIN));
                    }
                    break;
                case SUBSCRIBER:
                default:
                    logger.error("Unexpected chat role. Skip." + chatRole);
            }
        });
    }

    public void checkChatHistoryAndRemoveIfEmpty(Conference conference) {
        // Проверка того, что история чата пустая, имеет смысл только для чатов конференций
        Option<Chat> existingChatO = chatClient.getChats(conference.getDbId(), ChatType.CONFERENCE);
        if (!existingChatO.isPresent()) {
            logger.info("Remove empty chat for conference={}: chat not exists", conference.getConferenceId());
            return;
        }
        conferenceUserDao.findOwner(conference.getDbId()).map(
                owner -> chatClient.getUser(owner.getUid()).map(user -> {
                    logger.info("Remove empty chat for conference={}: chat owner was found", conference.getConferenceId());
                    if (chatClient.chatIsEmpty(conference.getDbId(), ChatType.CONFERENCE, user)) {
                        logger.info("Remove empty chat for conference={}: chat is empty", conference.getConferenceId());
                        chatClient.removeChatMembers(conference.getDbId(), ChatType.CONFERENCE);
                    }
                    return null;
                })
        );
    }
}
