package ru.yandex.chemodan.app.telemost.services;

import java.io.IOException;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.joda.time.Instant;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.telemost.exceptions.TelemostRuntimeException;
import ru.yandex.chemodan.app.telemost.repository.dao.ClientConfigurationDao;
import ru.yandex.chemodan.app.telemost.repository.model.ClientConfigurationDto;
import ru.yandex.chemodan.app.telemost.services.model.AdminClientConfiguration;
import ru.yandex.chemodan.app.telemost.services.model.ClientConfiguration;

public class ClientConfigurationService {

    private final ObjectMapper objectMapper;

    private final ClientConfigurationDao clientConfigurationDao;

    private final TransactionTemplate transactionTemplate;

    public ClientConfigurationService(ObjectMapper objectMapper, ClientConfigurationDao clientConfigurationDao,
            TransactionTemplate transactionTemplate)
    {
        this.objectMapper = objectMapper;
        this.clientConfigurationDao = clientConfigurationDao;
        this.transactionTemplate = transactionTemplate;
    }

    public AdminClientConfiguration getFormattedActualClientConfiguration() {
        Option<ClientConfigurationDto> actualClientConfigurationDtoO = clientConfigurationDao.getActualConfiguration();
        if (!actualClientConfigurationDtoO.isPresent()) {
            return new AdminClientConfiguration(serializeClientConfigurationToJson(new ClientConfiguration()), "");
        }
        ClientConfigurationDto clientConfigurationDto = actualClientConfigurationDtoO.get();
        return new AdminClientConfiguration(
                serializeClientConfigurationToJson(parseClientConfigurationFromJson(clientConfigurationDto.getValue())),
                clientConfigurationDto.getComment().getOrElse(""));
    }

    public void addNewClientConfiguration(String value, String createdByUser, Option<String> comment) {
        parseClientConfigurationFromJson(value);
        ClientConfigurationDto clientConfigurationDto =
                ClientConfigurationDto.createActual(Instant.now(), createdByUser, comment, value);
        transactionTemplate.execute(status -> saveActualClientConfiguration(clientConfigurationDto));
    }

    public Option<ClientConfigurationDto> getActualClientConfiguration() {
        return clientConfigurationDao.getActualConfiguration();
    }

    public MapF<String, Object> getV1ClientConfiguration() {
        return getClientConfigurationPart(ClientConfiguration::getV1);
    }

    public MapF<String, Object> getV2ClientConfiguration() {
        return getClientConfigurationPart(ClientConfiguration::getV2);
    }

    private MapF<String, Object> getClientConfigurationPart(Function<ClientConfiguration,
                Option<MapF<String, Object>>> partExtractor)
    {
        return clientConfigurationDao.getActualConfiguration().map(ClientConfigurationDto::getValue)
                .map(this::parseClientConfigurationFromJson).flatMapO(partExtractor).getOrElse(Cf.map());
    }

    private ClientConfigurationDto saveActualClientConfiguration(ClientConfigurationDto clientConfigurationDto) {
        clientConfigurationDao.expireActualConfiguration();
        return clientConfigurationDao.insertActualConfiguration(clientConfigurationDto);
    }

    private ClientConfiguration parseClientConfigurationFromJson(String json) {
        try {
            return objectMapper.readValue(json, ClientConfiguration.class);
        } catch (IOException e) {
            throw new TelemostRuntimeException(String.format("Invalid JSON '%s'", json), e);
        }
    }

    private String serializeClientConfigurationToJson(ClientConfiguration clientConfiguration) {
        try {
            return objectMapper.writeValueAsString(clientConfiguration);
        } catch (IOException e) {
            throw new TelemostRuntimeException(e);
        }
    }
}
