package ru.yandex.chemodan.app.telemost.services;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.telemost.appmessages.AppMessage;
import ru.yandex.chemodan.app.telemost.appmessages.AppMessageSender;
import ru.yandex.chemodan.app.telemost.appmessages.model.UserSource;
import ru.yandex.chemodan.app.telemost.appmessages.model.commands.Command;
import ru.yandex.chemodan.app.telemost.appmessages.model.commands.SendCommandRequest;
import ru.yandex.chemodan.app.telemost.appmessages.model.events.EventMessage;
import ru.yandex.chemodan.app.telemost.appmessages.model.events.RoleChangedEvent;
import ru.yandex.chemodan.app.telemost.exceptions.CommandNotAllowedException;
import ru.yandex.chemodan.app.telemost.exceptions.NoSuchUserInConference;
import ru.yandex.chemodan.app.telemost.exceptions.PeerNotFoundException;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferencePeerDao;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceUserDao;
import ru.yandex.chemodan.app.telemost.repository.dao.UserStateDtoDao;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceDto;
import ru.yandex.chemodan.app.telemost.repository.model.ConferencePeerDto;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceUserDto;
import ru.yandex.chemodan.app.telemost.repository.model.UserRole;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class CommandService {
    private static final Logger logger = LoggerFactory.getLogger(CommandService.class);

    private final ConferencePeerDao conferencePeerDao;
    private final UserStateDtoDao userStateDtoDao;
    private final AppMessageSender appMessageSender;
    private final ConferenceUserDao conferenceUserDao;
    private final ConferenceService conferenceService;
    private final ChatService chatService;

    public void sendCommand(ConferenceDto conference, String peer, PassportOrYaTeamUid uid, Command command) {
        conferencePeerDao.findPeerInConference(conference.getConferenceId(), peer).orElseThrow(PeerNotFoundException::new);
        checkIsAdmin(conference, uid);
        appMessageSender.sendMessageAsync(conference.getConferenceId(), new SendCommandRequest(new UserSource(uid), command), peer);
    }

    public void sendCommandToAll(ConferenceDto conference, PassportOrYaTeamUid uid, Command command, String excludePeer) {
        checkIsAdmin(conference, uid);
        appMessageSender.sendMessageToAllAsync(
                conference.getConferenceId(), new SendCommandRequest(new UserSource(uid), command), excludePeer);
    }

    public void setRole(ConferenceDto conferenceDto, PassportOrYaTeamUid senderUid, PassportOrYaTeamUid targetUid, UserRole newRole) {
        if (newRole.equals(UserRole.OWNER))
            throw new CommandNotAllowedException();
        if (targetUid.equals(senderUid))
            throw new CommandNotAllowedException();
        checkIsAdmin(conferenceDto, senderUid);
        UserRole currentRole = conferenceUserDao.findByConferenceAndUid(conferenceDto.getId(), targetUid)
                .map(ConferenceUserDto::getRole)
                .orElseThrow(NoSuchUserInConference::new);
        if (currentRole.equals(UserRole.OWNER)) {
            throw new CommandNotAllowedException();
        }
        if (currentRole.equals(newRole)) {
            return;
        }
        conferenceUserDao.upsert(conferenceDto.getId(), targetUid, newRole);
        ListF<ConferencePeerDto> peers = conferencePeerDao.findByConferenceAndUid(conferenceDto.getConferenceId(), targetUid);
        userStateDtoDao.incrementVersion(peers.map(ConferencePeerDto::getId));

        for (String targetPeerId : peers.map(ConferencePeerDto::getPeerId)) {
            AppMessage roleChangedEvent = new EventMessage(new UserSource(senderUid), new RoleChangedEvent(targetPeerId, newRole));
            appMessageSender.sendMessageToAll(conferenceDto.getConferenceId(), roleChangedEvent);
        }
        try {
            chatService.setRole(conferenceDto, targetUid, newRole.getChatRole());
        } catch (Exception e) {
            logger.error("Can't change role. Conference={}, uid={}, role={}",
                    conferenceDto.getConferenceId(), targetUid, newRole.getChatRole(), e);
        }
    }

    private void checkIsAdmin(ConferenceDto conference, PassportOrYaTeamUid uid) {
        conferenceService.ensureIsAdmin(conference, uid);
    }
}
