package ru.yandex.chemodan.app.telemost.services;

import java.net.URI;
import java.time.Instant;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.AllArgsConstructor;
import org.apache.http.HttpEntity;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceDto;
import ru.yandex.chemodan.util.http.HttpVoidResponseHandler;
import ru.yandex.chemodan.web.JacksonPojo;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author friendlyevil
 */
@AllArgsConstructor
public class ConferenceManagerClient {
    private static final Logger logger = LoggerFactory.getLogger(ConferenceManagerClient.class);
    private static final HttpVoidResponseHandler voidResponseHandler = new HttpVoidResponseHandler();

    private final HttpClient httpClient;
    private final URI baseUrl;
    private final int shardId;
    private final ObjectMapper objectMapper;
    private final boolean safeCall;

    public void saveConferenceMeta(ConferenceDto conference) {
        try {
            URI uri = UriBuilder.cons(baseUrl).appendPath("/api/v1/conference").build();
            HttpPost request = new HttpPost(uri);
            request.setEntity(ConferenceManagerRequestEntity.getEntity(objectMapper, conference, shardId));
            ApacheHttpClientUtils.execute(request, httpClient, voidResponseHandler);
        } catch (Exception e) {
            logger.warn("Error when saving conference meta to tcm", e);
            if (!safeCall) {
                throw e;
            }
        }
    }

    @AllArgsConstructor
    private static class ConferenceManagerRequestEntity extends DefaultObject implements JacksonPojo {
        @JsonProperty(value = "short_url_id")
        String shorUrlId;

        @JsonProperty(value = "conference_id")
        String conferenceId;

        @JsonProperty(value = "shard_id")
        int shardId;

        @JsonProperty("expired_at")
        Option<Instant> expiredAt;

        protected static HttpEntity getEntity(ObjectMapper objectMapper, ConferenceDto dto, int shardId) {
            try {
                return new StringEntity(objectMapper.writeValueAsString(
                        new ConferenceManagerRequestEntity(dto.getShortUrlId(), dto.getConferenceId(), shardId, Option.empty())),
                        ContentType.APPLICATION_JSON);
            } catch (JsonProcessingException e) {
                throw ExceptionUtils.translate(e);
            }
        }
    }
}
