package ru.yandex.chemodan.app.telemost.services;

import java.util.UUID;

import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.telemost.exceptions.ActiveSessionNotFoundTelemostException;
import ru.yandex.chemodan.app.telemost.exceptions.PeerNotFoundException;
import ru.yandex.chemodan.app.telemost.exceptions.SessionAlreadyDisconnectedTelemostException;
import ru.yandex.chemodan.app.telemost.exceptions.TooManyUsersTelemostException;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferencePeerDao;
import ru.yandex.chemodan.app.telemost.repository.dao.MediaSessionDao;
import ru.yandex.chemodan.app.telemost.repository.model.ApiVersion;
import ru.yandex.chemodan.app.telemost.repository.model.ConferencePeerDto;
import ru.yandex.chemodan.app.telemost.repository.model.MediaSessionDto;
import ru.yandex.chemodan.app.telemost.repository.model.MediaSessionWithPeerTokenDto;
import ru.yandex.chemodan.app.telemost.repository.model.UserByConferences;
import ru.yandex.chemodan.app.telemost.room.proto.MediatorOuterClass;
import ru.yandex.chemodan.app.telemost.room.proto.RoomGrpc;
import ru.yandex.chemodan.app.telemost.services.model.Conference;
import ru.yandex.chemodan.app.telemost.services.model.ConferenceParticipant;
import ru.yandex.chemodan.app.telemost.services.model.CreateSessionData;
import ru.yandex.chemodan.app.telemost.services.model.MediaSession;
import ru.yandex.chemodan.app.telemost.services.model.ParticipantsData;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.chemodan.app.telemost.services.model.User;
import ru.yandex.chemodan.app.telemost.services.model.UserDetails;
import ru.yandex.chemodan.app.telemost.util.UUIDUtils;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class ConferenceParticipantsService {

    private static final Logger logger = LoggerFactory.getLogger(ConferenceParticipantsService.class);

    private final DynamicProperty<String> defaultLanguage =
            new DynamicProperty<>("telemost.display_name.default_language", "ru");
    private final static String DEFAULT_USER_NAME = "Гость";
    private static final MapF<String, String> defaultNameTranslations = Cf.map("ru", DEFAULT_USER_NAME);

    private final ConferencePeerService conferencePeerService;
    private final ConferencePeerDao conferencePeerDao;
    private final RoomGrpc.RoomBlockingStub roomBlockingStub;
    private final MediaSessionDao mediaSessionDao;
    private final LimitsService limitsService;
    private final ParticipantIdGenerator participantIdGenerator;

    public ListF<UserByConferences> findByUid(String uid) {
        return conferencePeerDao.findByUid(uid);
    }

    public ListF<ConferenceParticipant> findConferenceUsers(Conference conference, ListF<String> peersIds) {
        ListF<ConferencePeerDto> conferencePeers = conferencePeerDao.findPeersInConference(conference.getConferenceId(), peersIds);
        Function0<MapF<String, User>> passportUsersProvider = ((Function0<MapF<String, User>>)
                () -> getPassportUsers(conferencePeers)).memoize();

        return conferencePeers.map(u ->
                new ConferenceParticipant(u,
                        () -> u.getUid().flatMapO(uid -> passportUsersProvider.apply().getO(uid)),
                        this::buildDisplayName));
    }

    public ConferenceParticipant findConferenceParticipant(Conference conference, String peerId) {
        ListF<ConferenceParticipant> conferenceUsers = findConferenceUsers(conference, Cf.list(peerId));
        if (conferenceUsers.size() > 1) {
            throw new IllegalStateException();
        }

        if (conferenceUsers.isEmpty()) {
            throw new PeerNotFoundException();
        }

        return conferenceUsers.single();
    }

    public ParticipantsData getParticipantsData(Conference conference) {
        return new ParticipantsData(Cf.toSet(getActivePeerIdsForConference(conference)));
    }

    public MediaSession createMediaSessionForUser(CreateSessionData createSessionData, ParticipantType participantType) {
        Option<MediaSessionDto> activeMediaSession = createSessionData.getClientInstanceId()
                .flatMapO(clientInstanceId ->
                        mediaSessionDao.getActiveMediaSession(createSessionData.getConference().getDbId(), clientInstanceId,
                                createSessionData.getUidAsString()));
        activeMediaSession.ifPresent(this::deactivateSession);
        String sessionId = UUID.randomUUID().toString();
        return activeMediaSession
                .map(session -> createSessionForExistingPeer(createSessionData.withDeactivatedSession(session)
                        .withSessionId(sessionId)))
                .getOrElse(() -> createSessionAndPeer(createSessionData.withSessionId(sessionId), participantType));
    }

    public MediaSession createSessionAndPeer(CreateSessionData createSessionData, ParticipantType participantType)
    {
        Option<String> clientInstanceId = createSessionData.getClientInstanceId();
        Conference conference = createSessionData.getConference();
        String peerId = clientInstanceId
                .flatMapO(clientInstanceIdValue -> conferencePeerDao.findUserByConferenceAndUidAndClientInstanceId(conference.getDbId(),
                        clientInstanceIdValue, createSessionData.getUidAsString()))
                .map(ConferencePeerDto::getPeerId)
                .getOrElse(() -> participantIdGenerator.createParticipantId(conference, clientInstanceId,
                        createSessionData.getUidAsString()));
        switch (participantType) {
            case TRANSLATOR:
                if (!peerId.startsWith(PeerPrefix.TRANSLATOR.getValue())) {
                    peerId = PeerPrefix.TRANSLATOR.getValue() + peerId;
                }
            default:
                break;
        }
        checkParticipantsCountLimits(peerId, createSessionData.getParticipantsData(), conference);
        ConferencePeerDto userDto = mergeByPeerId(createSessionData.getUser(), peerId, conference, createSessionData.getUserDetails());
        String sessionId = createSessionData.getSessionId().getOrThrow(IllegalStateException::new);
        MediaSessionDto session = MediaSessionDto.createActiveForInsert(sessionId, userDto.getId(), clientInstanceId);
        if (!mediaSessionDao.insert(session).isPresent()) {
            throw new SessionAlreadyDisconnectedTelemostException();
        }
        createSessionData.getPeerIdProcessor().accept(peerId);
        return new MediaSession(sessionId, peerId, userDto.getPeerToken().getOrThrow(IllegalStateException::new));
    }

    public void deactivateSession(MediaSessionDto mediaSessionDto) {
        int deactivatedSessionsCount = mediaSessionDao.deactivateSession(mediaSessionDto);
        if (deactivatedSessionsCount == 0) {
            throw new SessionAlreadyDisconnectedTelemostException();
        }
        logger.info("deactivated session={} count={}", mediaSessionDto, deactivatedSessionsCount);
    }

    public Option<MediaSessionDto> removeMVP1UserFromConference(Conference conference, String peerId) {
        Option<ConferencePeerDto> peerToDeactivateO = conferencePeerDao.findActiveMVP1Peer(conference.getDbId(), peerId);
        if (!peerToDeactivateO.isPresent()) {
            logger.info("No active MVP1 user roomId={} peerId={}", conference.getConferenceId(), peerId);
            return Option.empty();
        }
        ConferencePeerDto userToDeactivate = peerToDeactivateO.get();
        MediaSessionDto deactivatingSession = MediaSessionDto.createDeactivatingMVP1Session(userToDeactivate.getId());
        return mediaSessionDao.tryToInsert(deactivatingSession);
    }

    public void removeParticipant(Conference conference, String userId, String sessionId) {
        try {
            roomBlockingStub.removeParticipant(MediatorOuterClass.RemoveParticipantRequest.newBuilder()
                    .setRoomId(conference.getRoomId()).setParticipantId(userId).setSessionId(sessionId).build());
        } catch (StatusRuntimeException e) {
            if (!Status.NOT_FOUND.getCode().equals(e.getStatus().getCode())) {
                throw e;
            }
        }
    }

    public String mergeByPeerId(String peerId, Conference conference, Option<String> displayName,
            Option<String> language, ApiVersion apiVersion)
    {
        ConferencePeerDto userDto = new ConferencePeerDto(Option.empty(), peerId, Option.empty(), conference.getDbId(),
                getDisplayName(displayName, language), Option.of(apiVersion), Option.of(UUIDUtils.generateUUIDHex()));
        userDto = conferencePeerDao.mergeByPeerId(userDto);
        return userDto.getPeerId();
    }

    public MediaSessionWithPeerTokenDto findMediaSession(Conference conference, String userId,
            String mediaSessionId) {
        return mediaSessionDao.getMediaSessionForUserAndConference(conference.getDbId(), userId, mediaSessionId)
                .getOrThrow(ActiveSessionNotFoundTelemostException::new);
    }

    private ListF<String> getActivePeerIdsForConference(Conference conference) {
        Option<MediatorOuterClass.GetParticipantsResponse> participantsResponse = Option.empty();
        try {
           participantsResponse = Option.of(
                   roomBlockingStub.getParticipants(MediatorOuterClass.GetParticipantsRequest.newBuilder()
                           .setRoomId(conference.getRoomId()).build())
           );
        } catch (StatusRuntimeException e) {
            if (!Status.NOT_FOUND.getCode().equals(e.getStatus().getCode())) {
                throw e;
            }
        }
        return Cf.x(participantsResponse
                .map(MediatorOuterClass.GetParticipantsResponse::getParticipantsList).getOrElse(Cf::list))
                .filter(participant -> !participant.getIsRemoved())
                .map(MediatorOuterClass.Participant::getParticipantId);
    }

    private MediaSession createSessionForExistingPeer(CreateSessionData createSessionData) {
        MediaSessionDto deactivatedSession = createSessionData.getDeactivatedSession()
                .getOrThrow(IllegalStateException::new);
        String sessionId = createSessionData.getSessionId().getOrThrow(IllegalStateException::new);
        MediaSessionDto newActiveSession = deactivatedSession.newActiveSession(sessionId);
        UserDetails userDetails = createSessionData.getUserDetails();
        ConferencePeerDto userDto = mergeByDbUserId(newActiveSession.getUserId(), userDetails.getDisplayName(),
                userDetails.getLanguage(), userDetails.getApiVersion(), createSessionData.getUser());
        Conference conference = createSessionData.getConference();
        checkParticipantsCountLimits(userDto.getPeerId(), createSessionData.getParticipantsData(), conference);
        if (!mediaSessionDao.insert(newActiveSession).isPresent()) {
            throw new SessionAlreadyDisconnectedTelemostException();
        }
        createSessionData.getPeerIdProcessor().accept(userDto.getPeerId());
        return new MediaSession(newActiveSession.getMediaSessionId(), userDto.getPeerId(),
                userDto.getPeerToken().getOrThrow(IllegalStateException::new));
    }

    private void checkParticipantsCountLimits(String peerId, ParticipantsData participantsData, Conference conference) {
        if (participantsData.getParticipantIds().containsTs(peerId)) {
            logger.info("Peer {} is already in conference", peerId);
            return;
        }
        Option<Integer> maxPeersInRoomCount = limitsService.getParticipantsLimitValue(conference);
        if (maxPeersInRoomCount.map(count -> participantsData.getParticipantIds().size() >= count).getOrElse(Boolean.FALSE)) {
            throw new TooManyUsersTelemostException(maxPeersInRoomCount.get());
        }
    }

    private String buildDisplayName(Option<User> registeredUserInfo) {
        Option<String> displayName = registeredUserInfo.flatMapO(User::getDisplayName);

        return getDisplayName(displayName, Option.of(defaultLanguage.get()));
    }

    private MapF<String, User> getPassportUsers(ListF<ConferencePeerDto> dtoUsers) {
        ListF<PassportOrYaTeamUid> uids = dtoUsers.flatMap(ConferencePeerDto::getUid).map(PassportOrYaTeamUid::parseUid);
        return conferencePeerService.findUsers(uids).toMapMappingToKey(u -> u.getUid().asString());
    }


    private ConferencePeerDto mergeByPeerId(Option<User> user, String peerId, Conference conference,
            UserDetails userDetails)
    {
        Option<String> displayName = userDetails.getDisplayName();
        if (user.isPresent() && user.get().getUid().isPassportUid()) {
            //для зарегестрированных пользователей не учитываем что передали, берем из паспорта
            displayName = user.get().getDisplayName();
        }

        ConferencePeerDto userDto = new ConferencePeerDto(Option.empty(), peerId,
                user.map(User::getUid).map(PassportOrYaTeamUid::asString), conference.getDbId(),
                getDisplayName(displayName, userDetails.getLanguage()),
                Option.of(userDetails.getApiVersion()), Option.of(UUIDUtils.generateUUIDHex()));
        return conferencePeerDao.mergeByPeerId(userDto);
    }

    public ConferencePeerDto mergeByDbUserId(UUID dbUserId, Option<String> displayName,
            Option<String> language, ApiVersion apiVersion, Option<User> user)
    {
        if (user.isPresent() && user.get().getUid().isPassportUid()) {
            //для зарегестрированных пользователей не учитываем что передали, берем из паспорта
            displayName = user.get().getDisplayName();
        }

        return conferencePeerDao.updateById(dbUserId, getDisplayName(displayName, language), apiVersion);
    }

    public String getDisplayName(Option<String> displayName, Option<String> languageO) {
        if (displayName.isPresent() && !StringUtils.isBlank(displayName.get())) {
            return displayName.get();
        }
        String language = languageO.isPresent() && !StringUtils.isBlank(languageO.get()) ?
                languageO.get() : defaultLanguage.get();

        return defaultNameTranslations.getO(language).getOrElse(DEFAULT_USER_NAME);
    }
}
