package ru.yandex.chemodan.app.telemost.services;

import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTimeZone;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.chemodan.app.telemost.services.model.User;
import ru.yandex.chemodan.util.exception.NotFoundException;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.protocol.request.params.AliasesParameterValue;
import ru.yandex.inside.passport.blackbox2.protocol.request.params.EmailsParameterValue;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxAbstractResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxAliases;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxAvatar;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxDbFields;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxDisplayName;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.net.LocalhostUtils;
import ru.yandex.misc.time.MoscowTime;

public class ConferencePeerService {

    private static final Logger logger = LoggerFactory.getLogger(ConferencePeerService.class);

    public static final ListF<Integer> MAIL_PRO_ENABLED_ATTRIBUTES = Cf.list(197, 198);

    public static final ListF<Integer> MAIL_PRO_DISABLED_ATTRIBUTES = Cf.list(199);

    private static final ListF<Integer> MAIL_PRO_ATTRIBUTES = MAIL_PRO_ENABLED_ATTRIBUTES.plus(MAIL_PRO_DISABLED_ATTRIBUTES);

    public static final String ENABLE_MAIL_PRO_ATTRIBUTE_VALUE = "1";

    private final Blackbox2 blackbox2;
    /**
     * @see <a href="https://docs.yandex-team.ru/blackbox/methods/userinfo#response_format">docs</a>
     */
    private final DynamicProperty<String> avatarsSize =
            new DynamicProperty<>("telemost.avatars-size", "islands-200");
    /**
     *
     */
    private final DynamicProperty<Boolean> useDisplayName =
            new DynamicProperty<>("telemost.use-display-name", false);
    /**
     * Для тестового окружения: https://avatars.mdst.yandex.net/get-yapic/
     * Для продового окружения: https://avatars.mds.yandex.net/get-yapic/
     */
    private final String avatarUrlPrefix;

    public ConferencePeerService(Blackbox2 blackbox2, String avatarUrlPrefix) {
        this.blackbox2 = blackbox2;
        this.avatarUrlPrefix = avatarUrlPrefix;
    }

    public ListF<User> findUsers(ListF<PassportOrYaTeamUid> uids) {
        if (uids.isEmpty()) {
            return Cf.list();
        }
        ListF<PassportUid> uidsToCheckInBb = uids.filter(PassportOrYaTeamUid::isPassportUid)
                .map(PassportOrYaTeamUid::getPassportUid).stableUnique();
        ListF<User> result = Cf.arrayList();
        if (!uidsToCheckInBb.isEmpty()) {
            MapF<PassportUid, BlackboxAbstractResponse> response =
                    blackbox2.query().userInfoBulk(LocalhostUtils.localAddress(),
                            uidsToCheckInBb,
                            Cf.list(BlackboxDbFields.FIRSTNAME, BlackboxDbFields.LASTNAME),
                            MAIL_PRO_ATTRIBUTES,
                            Option.of(EmailsParameterValue.GET_ALL),
                            Option.of(AliasesParameterValue.ALL), true, Option.empty(), Option.empty());
            result.addAll(response.values()
                    .map(BlackboxAbstractResponse::getOrThrow)
                    .filterMap(this::mapUser));
        }
        return result.unmodifiable().plus(uids.filter(uid -> !uid.isPassportUid()).map(this::createNonPassportUser));
    }

    public User findUser(PassportOrYaTeamUid uid) {
        if (!uid.isPassportUid()) {
            return createNonPassportUser(uid);
        }
        BlackboxCorrectResponse response = blackbox2.query().userInfo(
                LocalhostUtils.localAddress(),
                Option.of(uid.getPassportUid()),
                Option.empty(), Option.empty(),
                Cf.list(BlackboxDbFields.FIRSTNAME, BlackboxDbFields.LASTNAME),
                MAIL_PRO_ATTRIBUTES,
                Option.of(EmailsParameterValue.GET_ALL),
                Option.of(AliasesParameterValue.ALL),
                true,
                Option.empty(),
                false,
                true
        );

        return mapUser(response).getOrThrow(() -> new NotFoundException("User not found"));
    }

    private User createNonPassportUser(PassportOrYaTeamUid uid) {
        return new User(uid, Option.empty(), Option.empty(), Option.empty(), uid.isYaTeamUid(), false,
                MoscowTime.TZ, "ru");
    }

    private Option<User> mapUser(BlackboxCorrectResponse response) {
        if (!response.getUid().isPresent()) {
            return Option.empty();
        }
        User user = new User(response.getUid().map(PassportOrYaTeamUid::passportUid).get(), getDisplayName(response),
                getAvatarUrl(response), getIsDefaultAvatar(response), isStaffUser(response), isMailProUser(response),
                getTimezone(response), getLanguage(response));
        logger.info("Map blackbox user data to {}", user);

        return Option.of(user);
    }

    private boolean isStaffUser(BlackboxCorrectResponse response) {
        return response.getAliases().getO(BlackboxAliases.YANDEXOID).isPresent();
    }

    private boolean isMailProUser(BlackboxCorrectResponse response) {
        return !MAIL_PRO_DISABLED_ATTRIBUTES.exists(attribute -> response.getAttributes().getO(attribute).isPresent()) &&
                MAIL_PRO_ENABLED_ATTRIBUTES.filter(response.getAttributes()::containsKeyTs)
                .filter(attribute -> ENABLE_MAIL_PRO_ATTRIBUTE_VALUE.equals(response.getAttributes().getTs(attribute)))
                .isNotEmpty();
    }

    private Option<String> getAvatarUrl(BlackboxCorrectResponse response) {
        Option<BlackboxDisplayName> displayName = response.getDisplayName();

        Option<String> avatarId = displayName.filterMap(BlackboxDisplayName::getAvatar).map(BlackboxAvatar::getDefaultAvatarId);
        return avatarId.map(s -> avatarUrlPrefix + s + "/" + avatarsSize.get());
    }

    private Option<Boolean> getIsDefaultAvatar(BlackboxCorrectResponse response) {
        Option<BlackboxDisplayName> displayName = response.getDisplayName();

        return displayName.filterMap(BlackboxDisplayName::getAvatar).map(BlackboxAvatar::isEmpty);
    }

    private Option<String> getDisplayName(BlackboxCorrectResponse response) {
        Option<String> firstName = response.getDbFields().getO(BlackboxDbFields.FIRSTNAME);
        if (!useDisplayName.get() && StringUtils.isNotBlank(firstName.getOrNull())) {
            return firstName;
        }

        Option<BlackboxDisplayName> displayName = response.getDisplayName();
        if (displayName.isPresent() && StringUtils.isNotBlank(displayName.get().getPublicName().getOrNull())) {
            return displayName.get().getPublicName();
        }

        return Option.empty();
    }

    private DateTimeZone getTimezone(BlackboxCorrectResponse response) {
        Option<String> ts = response.getDbFields().getO(BlackboxDbFields.TIMEZONE_DB_FIELD);
        if (ts.isPresent()) {
            return DateTimeZone.forID(ts.get());
        }
        return MoscowTime.TZ;
    }

    private String getLanguage(BlackboxCorrectResponse response) {
        Option<String> langO = response.getDbFields().getO(BlackboxDbFields.LANG);
        if (langO.isPresent()) {
            String l = langO.get();
            if (l.equals("ru") || l.equals("en") || l.equals("tr") || l.equals("uk")) {
                return l;
            }
        }
        Option<String> domainO = response.getDomain();
        if (domainO.isPresent()) {
            String[] c = domainO.get().split("\\.");
            String l = c[c.length - 1];
            if (l.equals("ru") || l.equals("en") || l.equals("tr") || l.equals("uk")) {
                return l;
            }
        }
        return "ru";
    }
}
