package ru.yandex.chemodan.app.telemost.services;

import java.util.UUID;
import java.util.function.Supplier;

import lombok.AllArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.telemost.exceptions.CommandNotAllowedException;
import ru.yandex.chemodan.app.telemost.exceptions.ConferenceActionNotAvailableException;
import ru.yandex.chemodan.app.telemost.exceptions.ConferenceLinkExpiredException;
import ru.yandex.chemodan.app.telemost.exceptions.ConferenceLinkNotComeException;
import ru.yandex.chemodan.app.telemost.exceptions.ConferenceNotAvailableTelemostException;
import ru.yandex.chemodan.app.telemost.exceptions.ConferenceNotFoundTelemostException;
import ru.yandex.chemodan.app.telemost.exceptions.ForbiddenAccessToStartBroadcast;
import ru.yandex.chemodan.app.telemost.exceptions.InvalidTranslatorTokenTelemostException;
import ru.yandex.chemodan.app.telemost.exceptions.YaTeamTokenAccessDeniedTelemostException;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceDtoDao;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceStateDao;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceUserDao;
import ru.yandex.chemodan.app.telemost.repository.dao.UserDao;
import ru.yandex.chemodan.app.telemost.repository.model.ApiVersion;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceDto;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceStateDto;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceUserDto;
import ru.yandex.chemodan.app.telemost.repository.model.UserDto;
import ru.yandex.chemodan.app.telemost.repository.model.UserRole;
import ru.yandex.chemodan.app.telemost.services.model.CheckUserTokenPolicy;
import ru.yandex.chemodan.app.telemost.services.model.Conference;
import ru.yandex.chemodan.app.telemost.services.model.ConferenceClientParameters;
import ru.yandex.chemodan.app.telemost.services.model.ConferenceUriData;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.chemodan.app.telemost.services.model.User;
import ru.yandex.chemodan.app.telemost.services.model.XMPPLimitType;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class ConferenceService {
    private static final Logger logger = LoggerFactory.getLogger(ConferenceService.class);

    private final Supplier<String> conferenceIdProvider;

    private final ConferenceDtoDao conferenceDao;

    private final ConferenceStateDao conferenceStateDao;

    private final UserDao userDao;

    private final ConferenceUserDao conferenceUserDao;

    private final CalendarService calendarService;

    private final TransactionTemplate transactionTemplate;

    private final int insertConferenceRetryCount;

    private final ConferenceUriService conferenceUriService;

    private final UserTokenService userTokenService;

    private final ListF<ConferenceCreator> conferenceCreators;

    private final LimitsService limitsService;

    private final StaffService staffService;

    private final BroadcastService broadcastService;
    private final BroadcastUriService broadcastUriService;

    private final ConferenceManagerClient conferenceManagerClient;
    private final PropertyManager propertyManager;

    private final boolean saveConferenceInfoToTcm;

    public boolean isStaffOwner(ConferenceDto conference) {
        boolean isStaffOwner = false;

        Option<String> ownerUid = findOwnerUid(conference);
        if (ownerUid.isPresent()) {
            isStaffOwner = PassportOrYaTeamUid.parseUid(ownerUid.get()).isYaTeamUid();
        }
        return isStaffOwner;
    }

    public ConferenceDto getConferenceByFullUri(String uri) {
        return findConferenceByUriId(getConferenceUriId(uri).getShortUrlId())
                .orElseThrow(ConferenceNotFoundTelemostException::new);
    }

    public void ensureValidTranslatorToken(ConferenceDto conference, String token) {
        if (!checkTranslatorToken(conference, Option.of(token))) {
            throw new InvalidTranslatorTokenTelemostException();
        }
    }

    /*
     * Recording Allowed
     */

    public boolean isLocalRecordingAllowed() {
        return propertyManager.getUseLocalRecordingAllowed();
    }

    public boolean isCloudRecordingAllowed() {
        return propertyManager.getUseCloudRecordingAllowed();
    }

    /*
     * Chats Allowed
     */

    private boolean isChatsAllowed(boolean isYaTeam, boolean isPermanent, boolean isStaffOwner) {
        boolean chatsAllowed = false;

        if (!isYaTeam) {
            if (isPermanent || isStaffOwner) {
                chatsAllowed = propertyManager.getUseYaTeamChatsAllowed();
            } else {
                chatsAllowed = propertyManager.getUseNonYaTeamChatsAllowed();
            }
        }

        logger.info("Is chats allowed: is ya-team={}, is permanent={}, is staff only={}, chats allowed={}, ya-team " +
                        "chats allowed={}, non ya-team chats allowed={}",
                isYaTeam, isPermanent, isStaffOwner, chatsAllowed, propertyManager.getUseYaTeamChatsAllowed(),
                propertyManager.getUseNonYaTeamChatsAllowed());

        return chatsAllowed;
    }

    public boolean isChatsAllowed(ConferenceClientParameters clientParameters) {
        logger.info("Is chats allowed: client parameters={}", clientParameters.toString());

        boolean isYaTeam = clientParameters.getStaffOnly().orElse(false) ||
                !clientParameters.getExternalMeeting().orElse(true);

        return isChatsAllowed(isYaTeam, clientParameters.getPermanent().orElse(false),
                clientParameters.getUser().isPresent() && clientParameters.getUser().get().isStaff());
    }

    public boolean isChatsAllowed(ConferenceDto conferenceDto, boolean isStaffOwner) {
        boolean isYaTeam = conferenceDto.isStaffOnly() || conferenceDto.isYaTeam();

        logger.info("Is chats allowed: is staff owner={}, is ya-team={}, is permanent={}",
                isStaffOwner, isYaTeam, conferenceDto.isPermanent());

        return isChatsAllowed(isYaTeam, conferenceDto.isPermanent(), isStaffOwner);
    }

    /*
     * Control Allowed
     */

    private boolean isControlAllowed(boolean isYaTeam, boolean isPermanent, boolean isStaffOwner) {
        boolean controlAllowed;

        if (isYaTeam || isPermanent || isStaffOwner) {
            controlAllowed = propertyManager.getUseYaTeamControlAllowed();
        } else {
            controlAllowed = propertyManager.getUseNonYaTeamControlAllowed();
        }

        logger.info("Is control allowed: is ya-team={}, is permanent={}, is staff only={}, control allowed={}, ya-team " +
                        "control allowed={}, non ya-team control allowed={}",
                isYaTeam, isPermanent, isStaffOwner, controlAllowed, propertyManager.getUseYaTeamControlAllowed(),
                propertyManager.getUseNonYaTeamControlAllowed());

        return controlAllowed;
    }

    public boolean isControlAllowed(ConferenceClientParameters clientParameters) {
        logger.info("Is control allowed: client parameters={}", clientParameters.toString());

        boolean isYaTeam = clientParameters.getStaffOnly().orElse(false) ||
                !clientParameters.getExternalMeeting().orElse(true);

        return isControlAllowed(isYaTeam, clientParameters.getPermanent().orElse(false),
                clientParameters.getUser().isPresent() && clientParameters.getUser().get().isStaff());
    }

    public boolean isControlAllowed(ConferenceDto conferenceDto, boolean isStaffOwner) {
        boolean isYaTeam = conferenceDto.isStaffOnly() || conferenceDto.isYaTeam();

        logger.info("Is control allowed: is staff owner={}, is ya-team={}, is permanent={}",
                isStaffOwner, isYaTeam, conferenceDto.isPermanent());

        return isControlAllowed(isYaTeam, conferenceDto.isPermanent(), isStaffOwner);
    }

    /*
     * Broadcast allowed
     */

    private boolean isBroadcastAllowed(boolean isYaTeam, boolean isPermanent, boolean isStaffOwner) {
        boolean broadcastAllowed;

        if (isYaTeam || isPermanent || isStaffOwner) {
            broadcastAllowed = propertyManager.getUseYaTeamBroadcastAllowed();
        } else {
            broadcastAllowed = propertyManager.getUseNonYaTeamBroadcastAllowed();
        }

        logger.info("Is broadcast allowed: is ya-team={}, is permanent={}, is staff only={}, broadcast allowed={}, ya-team " +
                        "broadcast allowed={}, non ya-team broadcast allowed={}",
                isYaTeam, isPermanent, isStaffOwner, broadcastAllowed, propertyManager.getUseYaTeamBroadcastAllowed(),
                propertyManager.getUseNonYaTeamBroadcastAllowed());

        return broadcastAllowed;
    }

    public boolean isBroadcastAllowed(ConferenceDto conferenceDto, boolean isStaffOwner) {
        boolean isYaTeam = conferenceDto.isStaffOnly() || conferenceDto.isYaTeam();

        logger.info("Is broadcasts allowed: is staff owner={}, is ya-team={}, is permanent={}",
                isStaffOwner, isYaTeam, conferenceDto.isPermanent());

        return isBroadcastAllowed(isYaTeam, conferenceDto.isPermanent(), isStaffOwner);
    }

    public boolean isBroadcastAllowed(boolean isStaff) {
        return isStaff ? propertyManager.getUseYaTeamBroadcastAllowed() : propertyManager.getUseNonYaTeamBroadcastAllowed();
    }

    /*
     * Broadcast Feature Enabled
     */

    public boolean isBroadcastFeatureEnabled(ConferenceDto conference) {
        return userDao.findByConferenceOwner(conference.getId()).map(UserDto::isBroadcastEnabled).orElse(false);
    }

    public boolean isBroadcastFeatureEnabled(PassportOrYaTeamUid uid) {
        return userDao.findByUid(uid).map(UserDto::isBroadcastEnabled).orElse(false);
    }

    /*
     * Service Methods
     */

    public Conference generateConference(ConferenceClientParameters clientParameters) {
        ConferenceDto conference = RetryUtils.retry(logger, insertConferenceRetryCount,
                () -> insertNewConference(conferenceIdProvider.get(), clientParameters));
        clientParameters.getUser()
                .map(User::getUid)
                .ifPresent(uid -> initConferenceOwner(conference, uid));
        RetryUtils.retry(logger, insertConferenceRetryCount,
                        () -> insertNewConferenceState(new ConferenceStateDto(
                                isLocalRecordingAllowed(),
                                isCloudRecordingAllowed(),
                                isChatsAllowed(conference, true),
                                Option.empty(),
                                isControlAllowed(conference, true),
                                isBroadcastAllowed(conference, true),
                                isBroadcastFeatureEnabled(conference),
                                Option.empty(),
                                Option.empty(),
                                conference.getId()))
        );
        return from(conference);
    }

    private void initConferenceOwner(ConferenceDto conference, PassportOrYaTeamUid uid) {
        // For ya-team uid also add linked staff user as conference admin
        RetryUtils.retry(logger, insertConferenceRetryCount, () -> {
            conferenceUserDao.upsert(conference.getId(), uid, UserRole.OWNER);
            if (uid.isYaTeamUid()) {
                staffService.resolveUid(Option.of(uid))
                        .map(PassportOrYaTeamUid::passportUid)
                        .map(staff -> conferenceUserDao.upsert(conference.getId(), staff, UserRole.ADMIN));
            }
        });
    }

    public Conference joinConference(Option<User> userO, String uri, Option<String> tvmUserTicket,
                                     Option<String> translatorToken) {
        return joinConference(userO, getConferenceUriId(uri), tvmUserTicket, translatorToken);
    }

    @SuppressWarnings("deprecation")
    public Conference joinConference(Option<User> userO, ConferenceUriData conferenceUriData,
                                     Option<String> tvmUserTicket, Option<String> translatorToken) {
        return transactionTemplate.execute(status -> {
            ConferenceDto conference = getConferenceDto(userO, conferenceUriData, CheckUserTokenPolicy.DEFAULT,
                    tvmUserTicket, true, translatorToken);
            userO.map(User::getUid).ifPresent(uid -> {
                // If owner is ya-team uid and there is no admins in conference join as ADMIN else as MEMBER
                Option<ConferenceUserDto> owner = conferenceUserDao.findOwner(conference.getId());
                if (owner.isPresent()) {
                    UserRole role = UserRole.MEMBER;
                    boolean adminPresents = conferenceUserDao
                            .findByConferenceAndRole(conference.getId(), UserRole.ADMIN).isNotEmpty();
                    boolean ownerIsYaTeam = owner.map(ConferenceUserDto::getUid)
                            .map(PassportOrYaTeamUid::parseUid)
                            .filter(PassportOrYaTeamUid::isYaTeamUid)
                            .isPresent();
                    // If admin is not present then try set admin from owner
                    if (ownerIsYaTeam && !adminPresents) {
                        Option<PassportUid> p = staffService.resolveUid(owner.get().getUid());
                        if (p.isPresent()) {
                            conferenceUserDao.insertIfNotExists(conference.getId(), PassportOrYaTeamUid.passportUid(p.get()), UserRole.ADMIN);
                        } else {
                            role = UserRole.ADMIN;
                        }
                    }
                    conferenceUserDao.insertIfNotExists(conference.getId(), uid, role);
                } else { // way to fix legacy conferences without owner
                    if (conference.getOwnerUid().isEmpty()) // for the sake of consistency for migration period
                        conferenceUserDao.upsert(conference.getId(), uid, UserRole.OWNER);
                    else {
                        try {
                            PassportOrYaTeamUid ownerUid = conference.getOwnerUid()
                                    .map(PassportOrYaTeamUid::parseUid).get();
                            conferenceUserDao.insertIfNotExists(conference.getId(), ownerUid, UserRole.OWNER);
                        } catch (Exception e) {
                            logger.error("Error while merging owner with uid {}", conference.getOwnerUid(), e);
                        }
                        conferenceUserDao.upsert(conference.getId(), uid, UserRole.MEMBER);
                    }
                }
            });
            return from(conference);
        });
    }

    public Conference findOrCreateConference(Option<User> uid, ConferenceUriData conferenceUriData,
                                             Option<String> tvmUserTicket)
    {
        ConferenceDto conference = getConferenceDto(uid, conferenceUriData, CheckUserTokenPolicy.DEFAULT,
                tvmUserTicket, false, Option.empty());
        return from(conference);
    }

    public Conference findOrCreateConferenceWithoutTokenActivating(Option<User> uid,
                                                                   ConferenceUriData conferenceUriData,
                                                                   Option<String> tvmUserTicket)
    {
        ConferenceDto conference = getConferenceDto(uid, conferenceUriData, CheckUserTokenPolicy.JUST_CHECK_EXIST,
                tvmUserTicket, true, Option.empty());
        return from(conference);
    }

    public Conference findOrCreateConferenceUnsafe(String conferenceId) {
        Option<ConferenceDto> conferenceO = conferenceDao.findByConferenceId(conferenceId);
        if (!conferenceO.isPresent()) {
            // создаем обычную конференцию
            return generateConference(conferenceId,
                    ConferenceClientParameters.builder()
                            .permanent(Option.of(Boolean.FALSE))
                            .staffOnly(Option.of(Boolean.FALSE))
                            .user(Option.empty())
                            .externalMeeting(Option.empty())
                            .eventId(Option.empty()).build());
        }
        ConferenceDto conference = conferenceO.get();
        return from(conference);
    }

    public Option<Conference> findConferenceUnsafe(String conferenceId) {
        return conferenceDao.findByConferenceId(conferenceId).map(this::from);
    }

    public ConferenceDto findById(UUID id) {
        return conferenceDao.findById(id);
    }

    public SetF<ApiVersion> getConferencePeersVersions(String conferenceId) {
        return conferenceDao.getConferencePeersVersions(conferenceId);
    }

    public ConferenceUriData getConferenceUriId(String conferenceUri) {
        return conferenceUriService.getConferenceUriData(conferenceUri);
    }

    public ConferenceUriData getYaTeamConferenceUriId(String conferenceUri) {
        return conferenceUriService.getYaTeamConferenceUriData(conferenceUri);
    }

    public String getYaTeamAuthorizationConferenceUri(Option<PassportOrYaTeamUid> uid,
            ConferenceUriData conferenceUriData)
    {
        Option<ConferenceDto> conferenceDtoO = conferenceDao.findByShortUriIdO(conferenceUriData.getShortUrlId());
        if (!conferenceDtoO.map(ConferenceDto::isYaTeam).orElse(Boolean.FALSE)) {
            throw new ConferenceNotFoundTelemostException();
        }
        ConferenceDto conferenceDto = conferenceDtoO.get();
        if (!isConferenceAcceptableByTtlParameter(conferenceDto)) {
            throw new ConferenceLinkExpiredException();
        }
        if (!uid.map(PassportOrYaTeamUid::isYaTeamUid).orElse(Boolean.FALSE)) {
            logger.info("The conference is unauthorized shortUriId={} uid={}", conferenceUriData.getShortUrlId(), uid);
            return conferenceUriService.buildConferenceAuthorizedUrl(conferenceDto, Option.empty());
        }
        String token = userTokenService.createUserToken(conferenceDto.getId());
        return conferenceUriService.buildConferenceAuthorizedUrl(conferenceDto, Option.of(token));
    }

    private boolean checkUserToken(Option<User> user, ConferenceUriData conferenceUriData,
            ConferenceDto conference, CheckUserTokenPolicy checkUserTokenPolicy)
    {
        return !conference.isYaTeam() ||
                checkUserTokenPolicy.checkToken(
                        userTokenService,
                        user,
                        conferenceUriData,
                        conference
                );
    }

    private boolean checkTranslatorToken(ConferenceDto conference, Option<String> translatorToken) {
        if (translatorToken.isEmpty()) {
            return true;
        }
        return broadcastService.checkTranslatorToken(conference.getId(), translatorToken.get());
    }

    public boolean isConferenceAcceptableByTtlParameter(ConferenceDto conference) {
        return conference.isPermanent() ||
                conference.getCreatedAt().plus(Duration.standardSeconds(propertyManager.getConferenceTtl())).isAfter(Instant.now());
    }

    private Conference generateConference(String conferenceId, ConferenceClientParameters clientParameters)
    {
        ConferenceDto conference = RetryUtils.retry(logger, insertConferenceRetryCount,
                () -> insertNewConference(conferenceId, clientParameters));

        return from(conference);
    }

    public ConferenceDto conferenceAcceptableByCalendarEventParameter(ConferenceDto conference,
                                                                     Option<PassportOrYaTeamUid> uid, Option<String> tvmUserTicket, boolean failOnCalendarError) {
        Option<String> ownerUid = findOwnerUid(conference);

        // Здесь заложена следующая логика:
        // - для корпоративного календаря никакие данные по событию не получаем, проверок не делаем
        // - для временных ссылок выполняется проверка на соответствие момента подключения времени начала и
        //   окончания встречи с учетом соответствующих лагов времени
        // - данные по событию не сохраняем в расчете на то, что все равно следует получать данные каждый раз когда
        //   они нужны поскольку выполняется проверка доступности события переданному юзер-тикету
        // - потенциально надо сохранять фактическое начало встречи в случае, если организатор начал
        //   встречу раньше (https://st.yandex-team.ru/TELEMOST-474)
        if (ownerUid.isPresent() && !PassportOrYaTeamUid.parseUid(ownerUid.get()).isYaTeamUid() && conference.getEventId().isPresent()) {
            logger.info("Check access to calendar event for conference id={} by user ticket permission",
                    conference.getShortUrlId());
            conference = calendarService.getCalendarEventData(conference, uid, tvmUserTicket, failOnCalendarError);
            if (conference.getStartEvent().isPresent()) {
                if (conference.getStartEvent().get().minus(Duration.standardSeconds(propertyManager.getConferenceTbl())).isAfterNow()) {
                    throw new ConferenceLinkNotComeException(conference.getStartEvent().get());
                } else if (conference.getStartEvent().get().plus(Duration.standardSeconds(propertyManager.getConferenceTtl())).isBeforeNow()) {
                    throw new ConferenceLinkExpiredException();
                }
            }
        } else {
            logger.info("No need to check access to calendar event data for conference id={}",
                    conference.getShortUrlId());
        }
        return conference;
    }

    public Option<ConferenceDto> findConferenceByUriId(String conferenceUri) {
        return conferenceDao.findByShortUriIdO(conferenceUri);
    }

    public Conference linkCalendarEvent(ConferenceUriData conferenceUriData, PassportOrYaTeamUid uid, String eventId) {
        ConferenceDto conferenceDto = findConferenceByUriId(conferenceUriData.getShortUrlId())
                .getOrThrow(ConferenceNotFoundTelemostException::new);
        Option<String> ownerUid = findOwnerUid(conferenceDto);
        if (!ownerUid.isPresent() || !ownerUid.get().equals(uid.asString())) {
            logger.info("Link calendar event for conference url={}, event id={} - can't link: conference owner={}, " +
                            "action uid={}",
                    conferenceUriData.getShortUrlId(), eventId, findOwnerUid(conferenceDto), uid.asString());
            throw new ConferenceActionNotAvailableException();
        }

        logger.info("Link calendar event for conference url={}, event id={} success",
                conferenceUriData.getShortUrlId(), eventId);
        return from(conferenceDao.addLinkToCalendarEvent(conferenceDto, eventId));
    }

    public void ensureIsAdmin(ConferenceDto conference, PassportOrYaTeamUid uid) {
        conferenceUserDao.findByConferenceAndUid(conference.getId(), uid)
                .map(ConferenceUserDto::getRole)
                .filter(UserRole::isAdmin)
                .orElseThrow(CommandNotAllowedException::new);
    }

    @NotNull
    private ConferenceDto getConferenceDto(Option<User> user, ConferenceUriData conferenceUriData,
                                           CheckUserTokenPolicy checkUserTokenPolicy, Option<String> tvmUserTicket,
                                           boolean getCalendarData, Option<String> translatorToken) {
        String shortUrlId = conferenceUriData.getShortUrlId();
        Option<ConferenceDto> conferenceO = conferenceDao.findByShortUriIdO(shortUrlId);
        if (!conferenceO.isPresent()) {
            throw new ConferenceNotFoundTelemostException();
        }
        ConferenceDto conference = conferenceO.get();

        if (getCalendarData) {

            // Get end check calendar event data
            conference = conferenceAcceptableByCalendarEventParameter(
                    conference, user.map(User::getUid), tvmUserTicket, true);

            // Check the expiration time of the link only if it has not already been checked above
            if (!conference.getEventId().isPresent()) {
                if (!isConferenceAcceptableByTtlParameter(conference)) {
                    throw new ConferenceLinkExpiredException();
                }
            }

        }

        if (!isConferenceAvailableForUser(user, conference)) {
            throw new ConferenceNotAvailableTelemostException(conference);
        }
        if (!checkUserToken(user, conferenceUriData, conference, checkUserTokenPolicy)) {
            throw new YaTeamTokenAccessDeniedTelemostException();
        }
        if (!checkTranslatorToken(conference, translatorToken)) {
            throw new InvalidTranslatorTokenTelemostException();
        }
        return conference;
    }

    protected boolean isConferenceAvailableForUser(Option<User> user, ConferenceDto conference) {
        return !conference.isStaffOnly() || user.map(limitsService::isStaffUser).orElse(Boolean.FALSE);
    }


    private ConferenceDto insertNewConference(String conferenceId, ConferenceClientParameters clientParameters) {
        XMPPLimitType limitType = limitsService.getLimitTypeForOwner(clientParameters.getUser());
        ConferenceClientParameters finalClientParameters = clientParameters.withLimitType(limitType);

        ConferenceDto conferenceDto = conferenceDao.insert(
                conferenceCreators.find(conferenceCreator -> conferenceCreator.isAcceptableFor(finalClientParameters))
                        .getOrThrow(IllegalStateException::new)
                        .createConferenceDto(conferenceId, finalClientParameters));

        if (saveConferenceInfoToTcm) {
            conferenceManagerClient.saveConferenceMeta(conferenceDto);
        }

        return conferenceDto;
    }

    public ConferenceDto findByBroadcastUri(String broadcastUri) {
        return conferenceDao.findByBroadcastKeyO(broadcastUriService.getBroadcastUriData(broadcastUri).getBroadcastKey())
                .map(x -> x).orElseThrow(ConferenceNotFoundTelemostException::new);
    }

    private ConferenceStateDto insertNewConferenceState(ConferenceStateDto state) {
        return conferenceStateDao.updateVersion(state);
    }

    public ConferenceStateDto incrementConferenceStateVersion(UUID conferenceId) {
        return conferenceStateDao.incrementVersion(conferenceId);
    }

    public void checkAdmin(ConferenceDto conferenceDto, PassportOrYaTeamUid uid) {
        Option<ConferenceUserDto> userO = conferenceUserDao.findByConferenceAndUid(conferenceDto.getId(), uid);
        logger.info("Check admin: conferenceDto={}, uid={}, user={}", conferenceDto, uid, userO);
        if (userO.isEmpty() || (userO.get().getRole() != UserRole.ADMIN && userO.get().getRole() != UserRole.OWNER)) {
            throw new ForbiddenAccessToStartBroadcast();
        }
    }

    private Option<String> findOwnerUid(ConferenceDto conferenceDto) {
        return conferenceUserDao.findOwner(conferenceDto.getId()).map(ConferenceUserDto::getUid);
    }

    private Conference from(ConferenceDto conferenceDto) {
        return new Conference(this, conferenceDto,
                conferenceUriService.buildConferenceUrl(conferenceDto), conferenceStateDao
        );
    }
}
