package ru.yandex.chemodan.app.telemost.services;

import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URLEncodedUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.exceptions.BadUriTelemostException;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceDto;
import ru.yandex.chemodan.app.telemost.services.model.ConferenceUriData;
import ru.yandex.chemodan.app.telemost.services.model.ConferenceUriGenerationData;

public class ConferenceUriService {
    private final Supplier<String> shortUrlIdProvider;

    private final Supplier<String> shortPermanentUrlIdProvider;

    private final String urlTemplate;

    private final Pattern shortUrlIdPattern;

    private final Pattern yaTeamUrlPattern;

    private final String uriHost;

    private final String yaTeamUriHost;

    public ConferenceUriService(
            Supplier<String> shortUrlIdProvider,
            Supplier<String> shortPermanentUrlIdProvider,
            String conferenceUriPattern,
            String urlTemplate,
            String yaTeamUrlPattern,
            String uriHost,
            String yaTeamUriHost)
    {
        this.shortUrlIdProvider = shortUrlIdProvider;
        this.shortPermanentUrlIdProvider = shortPermanentUrlIdProvider;
        this.shortUrlIdPattern = Pattern.compile(conferenceUriPattern);
        this.urlTemplate = urlTemplate;
        this.yaTeamUrlPattern = Pattern.compile(yaTeamUrlPattern);
        this.uriHost = uriHost;
        this.yaTeamUriHost = yaTeamUriHost;
    }

    public ConferenceUriData getConferenceUriData(String conferenceUri) {
        return getConferenceUriData(conferenceUri, shortUrlIdPattern);
    }

    public ConferenceUriData getYaTeamConferenceUriData(String conferenceUri) {
        return getConferenceUriData(conferenceUri, yaTeamUrlPattern);
    }

    private ConferenceUriData getConferenceUriData(String conferenceUri, Pattern pattern) {
        Matcher matcher = pattern.matcher(conferenceUri);
        if (!matcher.matches()) {
            throw new BadUriTelemostException(String.format("Bad uri format '%s'", conferenceUri));
        }
        URI uri = createUri(conferenceUri);
        String path = uri.getPath();
        String shortId = path.substring(path.lastIndexOf("/") + 1);
        ListF<NameValuePair> parameters = Cf.x(URLEncodedUtils.parse(uri.getQuery(), StandardCharsets.UTF_8));
        Option<String> ytToken = parameters.find(parameter -> "yt-token".equals(parameter.getName())).map(NameValuePair::getValue);
        return new ConferenceUriData(shortId, ytToken);
    }

    public String createShortUrlId(boolean isPermanent) {
        return isPermanent ? shortPermanentUrlIdProvider.get() : shortUrlIdProvider.get();
    }

    public String buildConferenceUrl(ConferenceDto conferenceDto) {
        return buildFullConferenceUrl(new ConferenceUriGenerationData(
                conferenceDto.getShortUrlId(),
                Option.empty(),
                getUriHostForConference(conferenceDto)));
    }

    public String buildConferenceAuthorizedUrl(ConferenceDto conferenceDto, Option<String> userToken) {
        return buildFullConferenceUrl(new ConferenceUriGenerationData(
                conferenceDto.getShortUrlId(),
                userToken,
                uriHost));
    }

    private String buildFullConferenceUrl(ConferenceUriGenerationData conferenceUriData) {
        return String.format(urlTemplate,
                conferenceUriData.getHost(),
                conferenceUriData.getShortUrlId() + conferenceUriData.getUserToken()
                        .map(userToken -> "?yt-token=" + userToken).getOrElse(""));
    }

    private URI createUri(String conferenceUri) {
        try {
            return new URI(conferenceUri);
        } catch (URISyntaxException e) {
            throw new BadUriTelemostException(String.format("Cannot parse the URI '%s'", conferenceUri));
        }
    }

    private String getUriHostForConference(ConferenceDto conferenceDto) {
        return conferenceDto.isYaTeam() ? this.yaTeamUriHost : this.uriHost;
    }
}
