package ru.yandex.chemodan.app.telemost.services;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.services.model.Conference;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.chemodan.app.telemost.services.model.User;
import ru.yandex.chemodan.app.telemost.services.model.XMPPLimitType;
import ru.yandex.commune.dynproperties.DynamicProperty;

public class LimitsService {

    private final DynamicProperty<String> defaultLimitType =
            new DynamicProperty<>("telemost-default-xmpp-limit-type", XMPPLimitType.STAFF.name());

    private final DynamicProperty<ListF<String>> xmppUnlimitedUids =
            new DynamicProperty<>("telemost-xmpp-unlimited-uids", Cf.list("376338392"));

    private final DynamicProperty<ListF<String>> xmppStaffOnlyUids =
            new DynamicProperty<>("telemost-xmpp-staff-only-uids", Cf.list("410033615"));

    private final DynamicProperty<Integer> defaultLimit = new DynamicProperty<>("telemost-limit-type-default-value", 35);

    private final DynamicProperty<Integer> staffLimit = new DynamicProperty<>("telemost-limit-type-staff-value", -1);

    private final DynamicProperty<Integer> unlimLimit = new DynamicProperty<>("telemost-limit-type-unlim-value", -1);

    private final MapF<XMPPLimitType, DynamicProperty<Integer>> limitValuesByType = Cf.map(
            XMPPLimitType.DEFAULT, defaultLimit,
            XMPPLimitType.STAFF, staffLimit,
            XMPPLimitType.UNLIM, unlimLimit
    );

    public XMPPLimitType getV1LimitType(XMPPLimitType sourceLimitType) {
        return sourceLimitType == XMPPLimitType.DEFAULT ? XMPPLimitType.valueOf(defaultLimitType.get()) : sourceLimitType;
    }

    public XMPPLimitType getLimitTypeForOwner(Option<User> user) {
        if (!user.isPresent()) {
            return XMPPLimitType.DEFAULT;
        }
        if (user.map(User::getUid).map(PassportOrYaTeamUid::asString)
                .map(xmppUnlimitedUids.get()::containsTs).getOrElse(Boolean.FALSE)) {
            return XMPPLimitType.UNLIM;
        }
        if (user.map(this::isStaffUser).get()) {
            return XMPPLimitType.STAFF;
        }
        return XMPPLimitType.DEFAULT;
    }

    public Option<Integer> getParticipantsLimitValue(Conference conference) {
        return Option.of(limitValuesByType.getO(conference.getLimitType()).getOrThrow(IllegalStateException::new))
                .map(DynamicProperty::get).filter(value -> value >= 0);
    }

    public boolean isStaffUser(User user) {
        if (user.isStaff()) {
            return true;
        }
        if (xmppStaffOnlyUids.get().containsTs(user.getUid().asString())) {
            return true;
        }
        return false;
    }
}
