package ru.yandex.chemodan.app.telemost.services;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferencePeerDao;
import ru.yandex.chemodan.app.telemost.repository.dao.UserStateDtoDao;
import ru.yandex.chemodan.app.telemost.repository.model.ConferencePeerDto;
import ru.yandex.chemodan.app.telemost.repository.model.UserBackData;
import ru.yandex.chemodan.app.telemost.repository.model.UserRole;
import ru.yandex.chemodan.app.telemost.repository.model.UserStateDto;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.chemodan.app.telemost.services.model.User;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class PeerStateService {
    private static final Logger logger = LoggerFactory.getLogger(PeerStateService.class);

    private final ConferencePeerService conferencePeerService;
    private final ObjectMapper objectMapper;
    private final UserStateDtoDao userStateDtoDao;
    private final ConferencePeerDao conferencePeerDao;

    public Option<UserBackData> getBackData(UserStateDto userStateDto) {
        return getBackData(userStateDto.getState());
    }

    public Option<UserBackData> getBackData(JsonNode state) {
        JsonNode backData = state.get(UserStateDto.BACK_DATA_FIELD);
        if (backData == null)
            return Option.empty();
        try {
            return Option.of(objectMapper.treeToValue(backData, UserBackData.class));
        } catch (JsonProcessingException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    public void setBackData(UserStateDto userState, UserBackData backData) {
        ((ObjectNode)userState.getState()).set(UserStateDto.BACK_DATA_FIELD, objectMapper.valueToTree(backData));
    }

    public void setRole(UserStateDto userState, UserRole role) {
        Option<UserBackData> oldData = getBackData(userState);
        UserBackData newData = oldData
                .map(data -> new UserBackData(data.getUid(), data.getDisplayName(),
                        data.getAvatarUrl(), data.getIsDefaultAvatar(), Option.of(role)))
                .orElse(new UserBackData(Option.empty(), Option.empty(), Option.empty(), Option.of(true),
                        Option.of(role)));
        setBackData(userState, newData);
    }

    public UserStateDto storePassportDataIfRequired(UserStateDto state, Option<PassportOrYaTeamUid> uid) {
        if (!state.isPassportDataRequired()) {
            return state;
        }
        UserBackData userBackData;
        if (uid.isPresent()) {
            try {
                User user = conferencePeerService.findUser(uid.get());
                userBackData = new UserBackData(user);
                logger.info("Get user info from passport for uid {}, user {}, data {}", uid, user, userBackData);
            } catch (Exception e) {
                logger.error("Can't get passport data for uid {}", uid, e);
                return state;
            }
        } else {
            Option<String> name = conferencePeerDao.findByIdO(state.getUserId()).map(ConferencePeerDto::getDisplayName);
            userBackData = new UserBackData(Option.empty(), name, Option.empty(), Option.of(true), Option.empty());
        }
        UserStateDto newState = userStateDtoDao.insertUserBackDataIntoState(state.getUserId(), userBackData).get();
        userStateDtoDao.setPassportDataRequired(state.getId(), false);
        logger.info("Update state for user id {}, uid {} on state {}", state.getUserId(), uid, newState);
        return newState;
    }
}
