package ru.yandex.chemodan.app.telemost.services;

import java.util.UUID;

import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.exceptions.InvalidPeerTokenTelemostException;
import ru.yandex.chemodan.app.telemost.repository.model.ApiVersion;
import ru.yandex.chemodan.app.telemost.repository.model.MediaSessionDto;
import ru.yandex.chemodan.app.telemost.repository.model.MediaSessionWithPeerTokenDto;
import ru.yandex.chemodan.app.telemost.room.RoomManager;
import ru.yandex.chemodan.app.telemost.services.model.Conference;
import ru.yandex.chemodan.app.telemost.services.model.CreateSessionData;
import ru.yandex.chemodan.app.telemost.services.model.MediaSession;
import ru.yandex.chemodan.app.telemost.services.model.ParticipantsData;
import ru.yandex.chemodan.app.telemost.services.model.RoomConnectionInfo;
import ru.yandex.chemodan.app.telemost.services.model.User;
import ru.yandex.chemodan.app.telemost.services.model.UserDetails;
import ru.yandex.chemodan.xiva.BasicXivaClient;
import ru.yandex.chemodan.xiva.XivaSecretSign;

public class RoomService {

    private final String serviceName;

    private final BasicXivaClient basicXivaClient;

    private final String xivaToken;

    private final ObjectMapper objectMapper;

    private final WebsocketUriService websocketUriService;

    private final ConferenceParticipantsService conferenceParticipantsService;

    private final RoomManager roomManager;

    private final ApiVersion apiVersion;

    private final BroadcastService broadcastService;

    public RoomService(String serviceName, BasicXivaClient basicXivaClient, WebsocketUriService websocketUriService,
                       ObjectMapper objectMapper, RoomManager roomManager, String xivaToken,
                       ConferenceParticipantsService conferenceParticipantsService, ApiVersion apiVersion,
                       BroadcastService broadcastService)
    {
        this.serviceName = serviceName;
        this.basicXivaClient = basicXivaClient;
        this.xivaToken = xivaToken;
        this.objectMapper = objectMapper;
        this.websocketUriService = websocketUriService;
        this.conferenceParticipantsService = conferenceParticipantsService;
        this.roomManager = roomManager;
        this.apiVersion = apiVersion;
        this.broadcastService = broadcastService;
    }

    public RoomConnectionInfo createRoomAndUserForConference(Conference conference, Option<User> user,
            Option<String> language, Option<String> clientInstanceId)
    {
        createRoom(conference);
        MediaSession mediaSession =
                conferenceParticipantsService.createSessionAndPeer(
                        CreateSessionData.builder()
                                .sessionId(Option.of(UUID.randomUUID().toString()))
                                .userDetails(new UserDetails(Option.empty(), language, apiVersion))
                                .conference(conference)
                                .peerIdProcessor(peerId -> addParticipantToConference(conference, peerId, ApiVersion.V2, user))
                                .user(user)
                                .clientInstanceId(clientInstanceId)
                                .participantsData(new ParticipantsData(Cf.set()))
                                .build(),
                        ParticipantType.USER
                );
        String userId = mediaSession.getPeerId();
        XivaSecretSign secretSign = basicXivaClient.getXivaSecretSign(Cf.list(serviceName), Cf.list(userId),
                xivaToken, objectMapper);
        String websocketUri =
                websocketUriService.buildWebsocketUri(secretSign, userId, mediaSession.getMediaSessionId());
        return new RoomConnectionInfo(userId, conference.getRoomId(), websocketUri, mediaSession.getMediaSessionId(),
                mediaSession.getPeerToken());
    }

    public void createRoom(Conference conference) {
        roomManager.createRoom(conference);
    }

    public RoomConnectionInfo joinToConference(Conference conference, Option<User> user,
            Option<String> displayName, Option<String> language, Option<String> clientInstanceId,
                                               ParticipantType participantType)
    {
        ParticipantsData participantsData = conferenceParticipantsService.getParticipantsData(conference);
        MediaSession mediaSession =
                conferenceParticipantsService.createMediaSessionForUser(
                        CreateSessionData.builder()
                                .sessionId(Option.empty())
                                .conference(conference)
                                .participantsData(participantsData)
                                .userDetails(new UserDetails(displayName, language, apiVersion))
                                .peerIdProcessor(peerId -> addParticipantToConference(conference, peerId, ApiVersion.V2, user))
                                .user(user)
                                .clientInstanceId(clientInstanceId)
                                .build(),
                        participantType);
        String peerId = mediaSession.getPeerId();

        if (participantType == ParticipantType.TRANSLATOR) {
            broadcastService.updateTranslatorPeerIfNeed(conference.getDbId(), peerId);
        }

        XivaSecretSign secretSign = basicXivaClient.getXivaSecretSign(Cf.list(serviceName),
                Cf.list(peerId), xivaToken, objectMapper);
        String websocketUri =
                websocketUriService.buildWebsocketUri(secretSign, peerId, mediaSession.getMediaSessionId());
        return new RoomConnectionInfo(peerId, conference.getRoomId(), websocketUri, mediaSession.getMediaSessionId(),
                mediaSession.getPeerToken());
    }

    public void disconnectUserFromConference(Conference conference, String peerId,
            String peerToken, String mediaSessionId)
    {
        MediaSessionWithPeerTokenDto mediaSessionWithPeerTokenDto =
                conferenceParticipantsService.findMediaSession(conference, peerId, mediaSessionId);
        if (!peerToken.equals(mediaSessionWithPeerTokenDto.getPeerToken())) {
            throw new InvalidPeerTokenTelemostException();
        }
        if (!mediaSessionWithPeerTokenDto.getMediaSessionDto().isActive()) {
            // the session has been already deactivated
            return;
        }
        conferenceParticipantsService.deactivateSession(mediaSessionWithPeerTokenDto.getMediaSessionDto());
        conferenceParticipantsService.removeParticipant(conference, peerId, mediaSessionId);
    }

    public void addPeerToConference(Conference conference, String peerId) {
        conferenceParticipantsService.mergeByPeerId(peerId, conference, Option.empty(), Option.empty(), ApiVersion.V1);
        addParticipantToConference(conference, peerId, ApiVersion.V1, Option.empty());
    }

    public void removePeerFromConference(Conference conference, String peerId) {
        Option<MediaSessionDto> deactivatingSession =
                conferenceParticipantsService.removeMVP1UserFromConference(conference, peerId);
        if (deactivatingSession.isPresent()) {
            roomManager.removeParticipant(conference, peerId);
        }
    }

    private void addParticipantToConference(Conference conference, String userId, ApiVersion peerVersion,
            Option<User> user)
    {
        roomManager.addParticipant(conference, userId, peerVersion == ApiVersion.V1, user);
    }

}
