package ru.yandex.chemodan.app.telemost.services;

import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;

import lombok.AllArgsConstructor;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.chemodan.app.telemost.appmessages.AppMessageSender;
import ru.yandex.chemodan.app.telemost.appmessages.model.BroadcastStatusChanged;
import ru.yandex.chemodan.app.telemost.exceptions.CommandNotAllowedException;
import ru.yandex.chemodan.app.telemost.exceptions.ConferenceLinkExpiredException;
import ru.yandex.chemodan.app.telemost.exceptions.NoSuchBroadcastCreatedException;
import ru.yandex.chemodan.app.telemost.exceptions.StreamNotStartedException;
import ru.yandex.chemodan.app.telemost.orchestrator.TranslatorsOrchestrator;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceStateDao;
import ru.yandex.chemodan.app.telemost.repository.dao.StreamDao;
import ru.yandex.chemodan.app.telemost.repository.model.BroadcastDto;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceDto;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceStateDto;
import ru.yandex.chemodan.app.telemost.repository.model.StreamDto;
import ru.yandex.chemodan.app.telemost.services.model.BroadcastAndConferenceUris;
import ru.yandex.chemodan.app.telemost.services.model.BroadcastUriData;
import ru.yandex.chemodan.app.telemost.services.model.BroadcastUserId;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.chemodan.app.telemost.services.model.Stream;
import ru.yandex.chemodan.app.telemost.services.model.StreamConnection;
import ru.yandex.chemodan.app.telemost.translator.TranslatorClient;
import ru.yandex.chemodan.app.telemost.ugcLive.UgcLiveClient;
import ru.yandex.chemodan.app.telemost.ugcLive.model.StreamSchedule;
import ru.yandex.chemodan.app.telemost.ugcLive.model.StreamScheduleItem;
import ru.yandex.chemodan.app.telemost.web.v2.model.BroadcastStateData;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.misc.ip.HostPort;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class StreamService {
    private static final Logger logger = LoggerFactory.getLogger(StreamService.class);

    private static final String STATE_READY = "ready";
    private static final String STATE_ONAIR = "onair";

    private final int insertStreamRetryCount;

    private final BroadcastService broadcastService;

    private final StreamDao streamDao;

    private final ConferenceService conferenceService;

    private final BroadcastUriService broadcastUriService;

    private final TranslatorsOrchestrator translatorsOrchestrator;

    private final TranslatorClient translatorClient;

    private final UgcLiveClient ugcLiveClient;

    private final UgcLiveService ugcLiveService;

    private final CalendarService calendarService;

    private final ConferenceStateDao conferenceStateDao;

    private final AppMessageSender appMessageSender;

    private final ConferencePeerService conferencePeerService;

    public Stream startStream(PassportOrYaTeamUid uid, BroadcastAndConferenceUris uris) {
        BroadcastDto broadcast = getVerifiedBroadcast(BroadcastUserId.user(uid), uris);

        // TODO: проверять не запущен ли уже стрим (по факту падает на ugcLiveClient.createEpisode)

        long lineId = getOrCreateUgcLiveLine(broadcast);

        UUID streamId = UUID.randomUUID();
        String sessionId = streamId.toString();
        HostPort translator = translatorsOrchestrator.acquireTranslator(sessionId);
        try {
            String ugcLiveSlug = ugcLiveClient.createEpisode(lineId, streamId.toString());
            try {
                String translatorToken = UUID.randomUUID().toString();
                String rtmpKey = ugcLiveClient.getLineInfo(lineId).getRtmpKey();

                StreamDto stream = new StreamDto(
                        streamId,
                        broadcast.getId().get(),
                        ugcLiveSlug,
                        Option.of(uid.asString()),
                        Option.of(Instant.now()),
                        Option.empty(),
                        Option.of(translator.toSerializedString()),
                        Option.of(translatorToken),
                        Option.of(rtmpKey),
                        Option.empty(),
                        Option.empty(),
                        getStreamUri(ugcLiveSlug)
                );
                executeUpdateWithRetries(() -> streamDao.insert(stream));

                conferenceStateDao.incrementVersion(broadcast.getConferenceId());

                sendAppMessage(broadcast.getConferenceId());

                translatorClient.start(translator, uris, rtmpKey, translatorToken);

                ugcLiveService.submitStreamPublishing(ugcLiveSlug);

                return new Stream(stream);

            } catch (Throwable e) {
                executeSuppressed(e, () -> executeUpdateWithRetries(() -> streamDao.delete(streamId)));
                executeSuppressed(e, () -> ugcLiveClient.deleteEpisode(ugcLiveSlug));
                throw e;
            }
        } catch (Throwable e) {
            executeSuppressed(e, () -> translatorsOrchestrator.releaseTranslator(sessionId));
            throw e;
        }
    }

    public String getStreamUri(String ugcLiveSlug) {
        return streamDao.getStreamUri(ugcLiveSlug);
    }

    public Stream stopStream(BroadcastUserId uid, BroadcastAndConferenceUris uris) {
        BroadcastDto broadcast = getVerifiedBroadcast(uid, uris);

        StreamDto stream = streamDao.findActiveByBroadcastKey(broadcast.getBroadcastKey())
                .orElseThrow(StreamNotStartedException::new);

        conferenceStateDao.incrementVersion(broadcast.getConferenceId());

        sendAppMessage(broadcast.getConferenceId());

        translatorClient.stop(HostPort.parse(stream.getTranslator().get()), uris, stream.getRtmpKey().get());

        translatorsOrchestrator.releaseTranslator(stream.getId().toString());

        ugcLiveService.stopOrCancelAndDeleteStream(stream.getUgcLiveSlug());

        StreamDto updatedStream = stream.withStoppedAt(Option.of(Instant.now()));

        executeUpdateWithRetries(() -> streamDao.update(updatedStream));

        return new Stream(updatedStream);
    }

    private CompletableFuture<?> sendAppMessage(UUID conferenceId) {
        Option<BroadcastStateData> broadcastStateDataO = Option.empty();

        Option<ConferenceStateDto> conferenceStateDtoO = conferenceStateDao.findState(conferenceId);

        if (!conferenceStateDtoO.isEmpty() && !conferenceStateDtoO.get().getBroadcast().isEmpty()) {
            broadcastStateDataO = Option.of(new BroadcastStateData(
                    conferenceStateDtoO.get().getBroadcast().get(),
                    conferenceStateDtoO.get().getStream()
            ));
        }

        return appMessageSender.sendMessageToAllAsync(
                conferenceService.findById(conferenceId).getConferenceId(), new BroadcastStatusChanged(broadcastStateDataO));
    }

    public StreamConnection getConnection(String broadcastUri) {
        BroadcastUriData broadcastUriData = broadcastUriService.getBroadcastUriData(broadcastUri);
        BroadcastDto broadcast = broadcastService.getByKey(broadcastUriData.getBroadcastKey())
                .orElseThrow(NoSuchBroadcastCreatedException::new);
        Option<String> displayNameO = conferencePeerService.findUser(
                PassportOrYaTeamUid.parseUid(broadcast.getCreatedBy())).getDisplayName();

        ConferenceDto conferenceDto = calendarService.getCalendarEventData(conferenceService.findById(broadcast.getConferenceId()), Option.empty(), Option.empty(), true);

        Option<Long> startEventTime = conferenceDto.getStartEvent().map(x -> x.getMillis() / 1000);
        Option<StreamDto> streamDtoO = streamDao.findActiveByBroadcastKey(broadcastUriData.getBroadcastKey());

        // Здесь заложена следующая логика:
        // - если конференция подвязана к событию календаря (а значит имеет время начала), то при подключении к стриму
        //   отдаем время начала и заголовок события
        // - если конференция не подвязана к событию календаря, то отдаем заголовок и описание, которые были получены
        //   при создании стрима
        if (startEventTime.isPresent()) {
            return streamDtoO.map(x -> new StreamConnection(Option.of(x.getStreamUri()),
                            Option.of(startEventTime.get()), conferenceDto.getEventCaption(), Option.empty(),
                            broadcast.getBroadcastChatPath(), broadcast.getStatus(),
                            displayNameO,
                            x.getStartedAt().map(t -> t.getMillis() / 1000)))
                    .orElse(new StreamConnection(Option.empty(),
                            Option.of(startEventTime.get()), conferenceDto.getEventCaption(), Option.empty(),
                            broadcast.getBroadcastChatPath(), broadcast.getStatus(),
                            displayNameO,
                            Option.empty()));
        }

        // Check the expiration time of the link
        if (!conferenceService.isConferenceAcceptableByTtlParameter(conferenceDto)) {
            throw new ConferenceLinkExpiredException();
        }

        return streamDtoO.map(x -> new StreamConnection(Option.of(x.getStreamUri()),
                        Option.empty(), broadcast.getCaption(), broadcast.getDescription(),
                        broadcast.getBroadcastChatPath(), broadcast.getStatus(),
                        displayNameO,
                        x.getStartedAt().map(t -> t.getMillis() / 1000)))
                .orElse(new StreamConnection(Option.empty(),
                        Option.empty(), broadcast.getCaption(), broadcast.getDescription(),
                        broadcast.getBroadcastChatPath(), broadcast.getStatus(),
                        displayNameO,
                        Option.empty()));
    }

    public Option<StreamDto> findActiveStreamByBroadcastKey(String broadcastKey) {
        return streamDao.findActiveByBroadcastKey(broadcastKey);
    }

    public void checkStreamsLive() {
        StreamSchedule streamSchedule = ugcLiveClient.getSchedule();
        MapF<String, StreamScheduleItem> streamScheduleMap = Cf.hashMap();

        for (StreamScheduleItem streamScheduleItem: streamSchedule.getStreamSchedule().filter(x -> Objects.equals(x.getStreamState(), "ready"))) {
            if (streamScheduleItem.getStreamState().equals(STATE_READY)
                    || streamScheduleItem.getStreamState().equals(STATE_ONAIR)) {
                streamScheduleMap.put(streamScheduleItem.getSlug(), streamScheduleItem);
            }
        }

        ListF<StreamDto> streams = streamDao.getActiveStreams();

        for (StreamDto streamDto: streams) {
            Option<StreamScheduleItem> streamScheduleItemO = streamScheduleMap.getO(streamDto.getUgcLiveSlug());
            if (streamScheduleItemO.isEmpty()) {
                logger.info("Stream slug={} not started. It will be restarting.", streamDto.getUgcLiveSlug());

                // TODO: restart stream and send appmessage
            }
        }
    }

    private BroadcastDto getVerifiedBroadcast(BroadcastUserId uid, BroadcastAndConferenceUris uris) {
        ConferenceDto conference = conferenceService.getConferenceByFullUri(uris.getConferenceUri());

        if (uid.getTranslatorToken().isPresent()) {
            conferenceService.ensureValidTranslatorToken(conference, uid.getTranslatorToken().get());
        } else if (uid.getUid().isPresent()) {
            conferenceService.ensureIsAdmin(conference, uid.getUid().get());
        } else {
            throw new CommandNotAllowedException();
        }
        BroadcastUriData broadcastUriData = broadcastUriService.getBroadcastUriData(uris.getBroadcastUri());

        BroadcastDto broadcast = broadcastService.getByKey(broadcastUriData.getBroadcastKey())
                .orElseThrow(NoSuchBroadcastCreatedException::new);

        if (!broadcast.getConferenceId().equals(conference.getId())) {
            throw new NoSuchBroadcastCreatedException();
        }
        return broadcast;
    }

    private long getOrCreateUgcLiveLine(BroadcastDto broadcastDto) {
        if (broadcastDto.getUgcLiveLineId().isPresent()) {
            return broadcastDto.getUgcLiveLineId().get();
        }
        long created = ugcLiveClient.createLine(broadcastDto.getId().toString());
        try {
            broadcastService.setUgcLiveLine(broadcastDto.getId().get(), created);
            return created;
        } catch (Throwable e) {
            ugcLiveClient.deleteLine(created);
            throw e;
        }
    }

    private void executeUpdateWithRetries(Function0V action) {
        RetryUtils.retry(logger, insertStreamRetryCount, action);
    }

    private void executeSuppressed(Throwable base, Function0V action) {
        try {
            action.apply();
        } catch (Throwable t) {
            base.addSuppressed(t);
        }
    }
}
