package ru.yandex.chemodan.app.telemost.services.model;

import java.util.Objects;
import java.util.UUID;

import lombok.AllArgsConstructor;
import lombok.EqualsAndHashCode;
import org.apache.commons.codec.digest.DigestUtils;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceStateDao;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceDto;
import ru.yandex.chemodan.app.telemost.services.ConferenceService;
import ru.yandex.misc.lang.DefaultObject;

@AllArgsConstructor
@EqualsAndHashCode(exclude = {"limitTypeF", "conferenceStateF"}, callSuper = false)
public class Conference extends DefaultObject {
    private final ConferenceDto conference;
    private final String uri;
    private final Function0<XMPPLimitType> limitTypeF;
    private final Function0<ConferenceState> conferenceStateF;

    public Conference(ConferenceService conferenceService, ConferenceDto conference, String uri, ConferenceStateDao conferenceStateDao) {
        this.conference = conference;
        this.uri = uri;
        this.limitTypeF = ((Function0<XMPPLimitType>) () ->
                Cf.x(XMPPLimitType.values())
                        .find(limitType -> limitType.name().equals(conference.getLimitType()))
                        .getOrThrow(IllegalStateException::new)
        ).memoize();
        this.conferenceStateF = ((Function0<ConferenceState>) () ->
                conferenceStateDao.findState(conference.getId())
                        .map(conferenceStateDao::from)
                        .orElse(new ConferenceState(conferenceService, conference))
        ).memoize();
    }

    private Conference(ConferenceDto conference, String uri, Function0<ConferenceState> conferenceStateF) {
        this.conference = conference;
        this.uri = uri;
        this.limitTypeF = ((Function0<XMPPLimitType>) () ->
                Cf.x(XMPPLimitType.values())
                        .find(limitType -> limitType.name().equals(conference.getLimitType()))
                        .getOrThrow(IllegalStateException::new)
        ).memoize();
        this.conferenceStateF = conferenceStateF;
    }

    public UUID getDbId() {
        return conference.getId();
    }

    public String getConferenceId() {
        return conference.getConferenceId();
    }

    public ConferenceDto getConferenceDto() {
        return conference;
    }

    public String getConferencePassword() {
        return conference.getConferencePassword();
    }

    public boolean isStaffOnly() {
        return conference.isStaffOnly();
    }

    public XMPPLimitType getLimitType() {
        return limitTypeF.apply();
    }

    public Instant getTimestamp() {
        return conference.getCreatedAt();
    }

    public String getUri() {
        return uri;
    }

    public ConferenceState getConferenceState() { return conferenceStateF.apply(); }

    public String getRoomId(){
        return conference.getConferenceId();
    }

    public String getSafeRoomId() {
        return DigestUtils.sha256Hex(getRoomId());
    }

    public String getShortUrlSHA256() {
        return DigestUtils.sha256Hex(getConferenceDto().getShortUrlId());
    }

    public Conference withLimitType(XMPPLimitType limitType) {
        return new Conference(conference.withLimitType(limitType.name()), uri, conferenceStateF);
    }

    public boolean isYaTeam() {
        return conference.isYaTeam();
    }

    public boolean isDifferentVersion(Long version) {
        return !Objects.equals(getConferenceState().getVersion(), version);
    }
}
