package ru.yandex.chemodan.app.telemost.services.model;

import java.util.UUID;

import lombok.EqualsAndHashCode;
import lombok.ToString;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.telemost.repository.model.ConferencePeerDto;
import ru.yandex.misc.lang.StringUtils;

@ToString
@EqualsAndHashCode(exclude = {"avatarUrlProvider", "isDefaultAvatarProvider", "displayNameProvider"})
public class ConferenceParticipant {

    private final ConferencePeerDto userDto;
    private final Function0<Option<String>> avatarUrlProvider;
    private final Function0<Option<Boolean>> isDefaultAvatarProvider;
    private final Function0<String> displayNameProvider;

    public ConferenceParticipant(ConferencePeerDto userDto, Function0<Option<User>> passportUserProvider,
            Function<Option<User>, String> backwardCompatibilityDisplayNameSupplier)
    {
        this.userDto = userDto;
        this.avatarUrlProvider = ((Function0<Option<String>>)
                () -> passportUserProvider.apply().flatMapO(User::getAvatarUrl)).memoize();
        this.isDefaultAvatarProvider = ((Function0<Option<Boolean>>)
                () -> passportUserProvider.apply().flatMapO(User::getIsDefaultAvatar)).memoize();
        this.displayNameProvider = ((Function0<String>) (() -> StringUtils.isNotBlank(userDto.getDisplayName())
                ? userDto.getDisplayName()
                : backwardCompatibilityDisplayNameSupplier.apply(passportUserProvider.apply()))).memoize();
    }

    public String getPeerId() {
        return userDto.getPeerId();
    }

    public UUID getDbUserId() {
        return userDto.getId();
    }

    public Option<PassportOrYaTeamUid> getUid() {
        return userDto.getUid().map(PassportOrYaTeamUid::parseUid);
    }

    public UUID getConferenceId() {
        return userDto.getConferenceId();
    }

    public String getDisplayName() {
        return displayNameProvider.apply();
    }

    public Option<String> getAvatarUrl() {
        return avatarUrlProvider.apply();
    }

    public Option<Boolean> getIsDefaultAvatar() {
        return isDefaultAvatarProvider.apply();
    }

}
