package ru.yandex.chemodan.app.telemost.services.model;

import java.util.concurrent.CompletableFuture;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.appmessages.AppMessageSender;
import ru.yandex.chemodan.app.telemost.appmessages.model.PassportAccountModification;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferencePeerDao;
import ru.yandex.chemodan.app.telemost.repository.dao.UserStateDtoDao;
import ru.yandex.chemodan.app.telemost.repository.model.ApiVersion;
import ru.yandex.chemodan.app.telemost.repository.model.UserBackData;
import ru.yandex.chemodan.app.telemost.repository.model.UserByConferences;
import ru.yandex.chemodan.app.telemost.repository.model.UserStateDto;
import ru.yandex.chemodan.app.telemost.services.ConferenceParticipantsService;
import ru.yandex.chemodan.app.telemost.services.ConferencePeerService;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.salr.logreader.LogListener;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class LbPassportConsumer implements LogListener {
    private static final Logger logger = LoggerFactory.getLogger(LbPassportConsumer.class);

    private final ConferenceParticipantsService conferenceParticipantsService;
    private final AppMessageSender appMessageSender;
    private final ConferencePeerService conferencePeerService;
    private final ConferencePeerDao conferencePeerDao;
    private final UserStateDtoDao userStateDtoDao;

    private final DynamicProperty<Boolean> handlePassportModification =
            new DynamicProperty<>("telemost.handle-passport-modification", true);

    public LbPassportConsumer(ConferenceParticipantsService conferenceParticipantsService,
                              AppMessageSender appMessageSender, ConferencePeerService conferencePeerService,
                              ConferencePeerDao conferencePeerDao, UserStateDtoDao userStateDtoDao) {
        this.conferenceParticipantsService = conferenceParticipantsService;
        this.appMessageSender = appMessageSender;
        this.conferencePeerService = conferencePeerService;
        this.conferencePeerDao = conferencePeerDao;
        this.userStateDtoDao = userStateDtoDao;
    }

    @Override
    public void processLogLine(String line) {

        if (!handlePassportModification.get()) {
            return;
        }

        // Get data of event from log
        Option<String> uidO = Option.empty();
        boolean isPresentConsumer = false;
        boolean isPresentEvent = false;
        boolean isPresentEntity = false;

        for (String s: line.split("\t")) {
            String[] v = s.split("=");
            if (v.length > 1) {
                if (v[0].equals("uid")) {
                    uidO = Option.of(v[1]);
                    if (isPresentConsumer && isPresentEvent) {
                        break;
                    }
                } else if (v[0].equals("consumer")) {
                    if (!v[1].equals("passport")) {
                        return;
                    }
                    isPresentConsumer = true;
                } else if (v[0].equals("event")) {
                    if (!v[1].equals("account_modification")) {
                        return;
                    }
                    isPresentEvent = true;
                } else if (v[0].equals("entity")) {
                    if (!v[1].equals("person.display_name") && !v[1].equals("person.default_avatar")) {
                        return;
                    }
                    isPresentEntity = true;
                }
            }
        }

        // Check needed data and send notification
        if (uidO.isPresent() && isPresentConsumer && isPresentEvent && isPresentEntity) {
            logger.info("From-Passport-LogBroker: " + line);
            updatePassportData(uidO.get());
        }
    }

    protected void updatePassportData(String uid) {
        PassportOrYaTeamUid passportUid = PassportOrYaTeamUid.parseUid(uid);
        if (!passportUid.isPassportUid()) {
            return;
        }
        userStateDtoDao.setPassportDataRequiredByUid(passportUid, true);
        UserBackData userBackData = new UserBackData(conferencePeerService.findUser(passportUid));

        conferenceParticipantsService.findByUid(uid).map(user -> {
            conferencePeerDao.updateById(
                        user.getUser().getId(),
                        conferenceParticipantsService.getDisplayName(userBackData.getDisplayName(), Option.empty()),
                        ApiVersion.V2);
            logger.info("Updated data by user back data for uid=" + uid +
                    ", conference_id=" + user.getConferenceId());

            Option<UserStateDto> userStateDto = userStateDtoDao.insertUserBackDataIntoState(user.getUser().getId(), userBackData);
                if (userStateDto.isPresent()) {
                    userStateDtoDao.setPassportDataRequired(userStateDto.get().getId(), false);
                    logger.info("Updated state by user back data for uid=" + uid +
                            ", conference_id=" + user.getConferenceId() +
                            " on " + userStateDto.get());
                } else {
                    logger.info("State by user back data for uid=" + uid +
                            ", conference_id=" + user.getConferenceId() +
                            " don't updated");
                }
                return sendNotification(uid, user);
            }
        );
    }

    public CompletableFuture<?> sendNotification(String uid, UserByConferences user) {
        CompletableFuture<?> future = appMessageSender.sendMessageToAllAsync(
                user.getConferenceId(),
                new PassportAccountModification(uid, user.getUser().getPeerId()));
        logger.info("Send notify message about passport account modification for uid={}, peer={}, conference={}",
                uid, user.getUser().getPeerId(), user.getConferenceId());
        return future;
    }
}
