package ru.yandex.chemodan.app.telemost.services.model;

import java.net.URI;
import java.util.concurrent.ThreadPoolExecutor;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.chemodan.app.telemost.appmessages.AppMessageSender;
import ru.yandex.chemodan.app.telemost.config.TelemostServicesContextConfiguration;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferencePeerDao;
import ru.yandex.chemodan.app.telemost.repository.dao.UserStateDtoDao;
import ru.yandex.chemodan.app.telemost.services.ConferenceParticipantsService;
import ru.yandex.chemodan.app.telemost.services.ConferencePeerService;
import ru.yandex.chemodan.logbroker.DiskLbConsumerContextConfiguration;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;
import ru.yandex.inside.logbroker.pull.consumer.AppCluster;
import ru.yandex.inside.logbroker.pull.consumer.LbConsumerAllDcSettingsProvider;
import ru.yandex.inside.logbroker.pull.consumer.LbConsumerSettings2;
import ru.yandex.inside.logbroker.pull.consumer.LbConsumerWorkerService;
import ru.yandex.inside.logbroker.pull.consumer.LbConsumers;
import ru.yandex.inside.logbroker.pull.consumer.LbLineListener;
import ru.yandex.inside.logbroker.pull.consumer.LbSimpleAsyncLineListener;
import ru.yandex.inside.logbroker.pull.consumer.SynchronousThreadPoolUtil;
import ru.yandex.inside.logbroker.pull.model.LbTopicFilter;
import ru.yandex.inside.logbroker.util.concurrent.GroupedExecutor;
import ru.yandex.inside.logbroker.util.concurrent.GroupedExecutorUtils;
import ru.yandex.misc.parse.CommaSeparated;

@Configuration
@Import({
        TelemostServicesContextConfiguration.class,
        DiskLbConsumerContextConfiguration.class,
})
public class LbPassportConsumerContextConfiguration {
    @Value("${logbroker.listener.executor.queue-timeout}")
    private Duration listenerQueueTimeout;

    @Value("${logbroker.listener.executor.total-thread-count}")
    private int listenerThreadsTotalCount;

    @Value("${logbroker.idents-to-exclude}")
    private CommaSeparated excludeIdents;

    @Value("${logbroker.passport.pull.logtype}")
    private String logtype;

    @Autowired
    @Qualifier("lbListenerExecutor")
    private ThreadPoolExecutor listenerExecutor;

    @Bean
    public LbPassportConsumer lbPassportConsumer(ConferenceParticipantsService conferenceParticipantsService,
                                                 AppMessageSender appMessageSender, ConferencePeerService conferencePeerService,
                                                 ConferencePeerDao conferencePeerDao, UserStateDtoDao userStateDtoDao)
    {
        return new LbPassportConsumer(conferenceParticipantsService, appMessageSender, conferencePeerService, conferencePeerDao,
                userStateDtoDao);
    }

    @Bean
    protected LbLineListener mainLbConsumerListener(ConferenceParticipantsService conferenceParticipantsService,
                                                    AppMessageSender appMessageSender, ConferencePeerService conferencePeerService,
                                                    ConferencePeerDao conferencePeerDao, UserStateDtoDao userStateDtoDao)
    {
        return new LbSimpleAsyncLineListener(
                lbPassportConsumer(conferenceParticipantsService, appMessageSender, conferencePeerService,
                        conferencePeerDao, userStateDtoDao),
                SynchronousThreadPoolUtil.listenerExecutor(listenerQueueTimeout, listenerThreadsTotalCount)
        );
    }

    protected LbConsumerSettings2 mainLbConsumerSettings(int listenerMaxQueueSize, URI lbBaseUrl, String lbClientId,
                                                         int lbSessionBatchSize, Duration lbSessionTimeout, Duration lbReadWait,
                                                         Duration minHardCommitInterval, Duration metaHostTimeout,
                                                         Duration dataHostTimeout)
    {
        return new LbPassportConsumerSettings2Impl("telemost-backend", listenerMaxQueueSize, lbBaseUrl,
                lbClientId, lbSessionBatchSize, lbSessionTimeout, lbReadWait, minHardCommitInterval,
                metaHostTimeout, dataHostTimeout,
                new LbTopicFilter()
                        .withIdentsToExclude(excludeIdents.list)
                        .withLogType(logtype)
        );
    }

    @Bean
    public LbSimpleAsyncLineListener lbConsumerListener(ConferenceParticipantsService conferenceParticipantsService,
                                                        AppMessageSender appMessageSender, ConferencePeerService conferencePeerService,
                                                        ConferencePeerDao conferencePeerDao, UserStateDtoDao userStateDtoDao)
    {
        return new LbSimpleAsyncLineListener(mainLbConsumerListener(conferenceParticipantsService, appMessageSender,
                conferencePeerService, conferencePeerDao, userStateDtoDao), listenerExecutor);
    }

    @Bean
    public GroupedExecutor<Long, Void> groupedExecutor(DynamicPropertyManager dynamicPropertyManager)
    {
        GroupedExecutor<Long, Void> groupedExecutor = new GroupedExecutor<>(listenerExecutor);
        GroupedExecutorUtils.bindToDynVars(groupedExecutor,
                ru.yandex.inside.logbroker.pull.DynamicVars.groupUidToOneTask,
                ru.yandex.inside.logbroker.pull.DynamicVars.maxUidQueueLength,
                dynamicPropertyManager
        );
        return groupedExecutor;
    }

    @Bean
    public LbConsumerWorkerService mainLbConsumerWorkerService(AppCluster appCluster,
                                                               @Qualifier("lbConsumerListener") LbLineListener lbLineListener,
                                                               @Value("${telemost-backend.logbroker.listener.max-queue-size}") int listenerMaxQueueSize,
                                                               @Value("${telemost-backend.logbroker.base-pull-url}") URI lbBaseUrl,
                                                               @Value("${telemost-backend.logbroker.client-id}") String lbClientId,
                                                               @Value("${telemost-backend.logbroker.session-batch-size}") int lbSessionBatchSize,
                                                               @Value("${telemost-backend.logbroker.session-timeout}") Duration lbSessionTimeout,
                                                               @Value("${telemost-backend.logbroker.read-wait}") Duration lbReadWait,
                                                               @Value("${telemost-backend.logbroker.min-hard-commit-interval}") Duration minHardCommitInterval,
                                                               @Value("${telemost-backend.logbroker.http.meta-host-timeout}") Duration metaHostTimeout,
                                                               @Value("${telemost-backend.logbroker.http.data-host-timeout}") Duration dataHostTimeout)
    {
        LbConsumerAllDcSettingsProvider lbConsumerSettings = new LbConsumerAllDcSettingsProvider(
                mainLbConsumerSettings(listenerMaxQueueSize, lbBaseUrl, lbClientId, lbSessionBatchSize,
                        lbSessionTimeout, lbReadWait, minHardCommitInterval, metaHostTimeout, dataHostTimeout));
        return new LbConsumerWorkerService(
                new LbConsumers(lbConsumerSettings, lbLineListener).forEachDcWithFixedNodeCount(appCluster)
        );
    }
}
