package ru.yandex.chemodan.app.telemost.services.model;

import java.util.regex.Pattern;

import lombok.ToString;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.exceptions.BadUidException;
import ru.yandex.inside.passport.PassportUid;

@ToString(doNotUseGetters = true)
public class PassportOrYaTeamUid {

    private static final Pattern YA_TEAM_UID_PATTERN = Pattern.compile("^yt:\\d+$");

    public static PassportOrYaTeamUid parseUid(String uid) {
        if (YA_TEAM_UID_PATTERN.matcher(uid).matches()) {
            return PassportOrYaTeamUid.yaTeamUid(uid);
        }
        try {
            return PassportOrYaTeamUid.passportUid(PassportUid.cons(Long.parseUnsignedLong(uid)));
        } catch (NumberFormatException e) {
            throw new BadUidException(String.format("Cannot parse the '%s' uid", uid));
        }
    }

    public static PassportOrYaTeamUid passportUid(PassportUid uid) {
        return new PassportOrYaTeamUid(Option.of(uid), Option.empty());
    }

    public static PassportOrYaTeamUid yaTeamUid(String yaTeamUid) {
        return new PassportOrYaTeamUid(Option.empty(), Option.of(yaTeamUid));
    }

    private final Option<PassportUid> passportUid;

    private final Option<String> yaTeamUid;

    private PassportOrYaTeamUid(Option<PassportUid> passportUid, Option<String> yaTeamUid) {
        this.passportUid = passportUid;
        this.yaTeamUid = yaTeamUid;
    }

    public boolean isPassportUid() {
        return passportUid.isPresent();
    }

    public boolean isYaTeamUid() {
        return yaTeamUid.isPresent();
    }

    public PassportUid getPassportUid() {
        return passportUid.getOrThrow(IllegalStateException::new);
    }

    public String getYaTeamUid() {
        return yaTeamUid.getOrThrow(IllegalStateException::new);
    }

    public String asString() {
        return isPassportUid() ? passportUid.map(PassportUid::toString).get() : yaTeamUid.get();
    }

    public String withoutPrefix() {
        return isPassportUid() ? passportUid.map(PassportUid::toString).get() : yaTeamUid.get().substring(3);
    }
}
