package ru.yandex.chemodan.app.telemost.translator;

import java.net.URI;

import lombok.AllArgsConstructor;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;

import ru.yandex.chemodan.app.telemost.services.model.BroadcastAndConferenceUris;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.http.apache.v4.VoidResponseHandler;
import ru.yandex.misc.ip.HostPort;

@AllArgsConstructor
public class TranslatorClientImpl implements TranslatorClient {

    private final HttpClient httpClient;

    @Override
    public void start(HostPort translator, BroadcastAndConferenceUris uris, String streamKey, String translatorToken) {
        URI uri = broadcastActionUrl(translator, uris, "start")
                .addParam("stream_key", streamKey)
                .addParam("translator_token", translatorToken)
                .build();

        HttpPut request = new HttpPut(uri);

        ApacheHttpClientUtils.execute(request, httpClient, new VoidResponseHandler());
    }

    @Override
    public void stop(HostPort translator, BroadcastAndConferenceUris uris, String streamKey) {
        URI uri = broadcastActionUrl(translator, uris, "stop")
                .addParam("stream_key", streamKey)
                .build();

        HttpPost request = new HttpPost(uri);

        ApacheHttpClientUtils.execute(request, httpClient, response -> {
            int code = response.getStatusLine().getStatusCode();

            if (code != 200 && code != 404 && code != 409) {
                throw new HttpException(code, "Got: " + response.getStatusLine());
            }
            return null;
        });
    }

    private UriBuilder broadcastActionUrl(HostPort translator, BroadcastAndConferenceUris uris, String action) {
        String baseUri = schemeHostPort(translator);
        String path = "/v2/conferences/" + UrlUtils.urlEncode(uris.getConferenceUri())
                + "/broadcast/" + UrlUtils.urlEncode(uris.getBroadcastUri())
                + "/" + action;

        // Can not use UriBuilder from scratch because of path encoding problems
        return UriBuilder.cons(baseUri + path);
    }

    private String schemeHostPort(HostPort translator) {
        switch (translator.getPort()) {
            case 80: return "http://" + translator.getHost().format();
            case 443: return "https://" + translator.getHost().format();
            default: return "http://" + translator.toSerializedString();
        }
    }
}
