package ru.yandex.chemodan.app.telemost.ugcLive;

import java.io.UncheckedIOException;
import java.net.URI;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.LongNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;
import lombok.AllArgsConstructor;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.ugcLive.model.LineInfo;
import ru.yandex.chemodan.app.telemost.ugcLive.model.StreamAction;
import ru.yandex.chemodan.app.telemost.ugcLive.model.StreamSchedule;
import ru.yandex.chemodan.app.telemost.ugcLive.model.StreamState;
import ru.yandex.chemodan.tvm2.UserTicketSupplier;
import ru.yandex.commune.json.jackson.StringEnumModule;
import ru.yandex.commune.json.jackson.bolts.BoltsModule;
import ru.yandex.inside.passport.tvm2.TvmHeaders;
import ru.yandex.inside.utils.JacksonResponseHandlerFactory;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.http.apache.v4.ReadOptionalStringResponseHandler;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.random.Random2;

/**
 * @link https://swagger-ui.yandex-team.ru/?url=https://vh.test.yandex.ru/v1/live/swagger.json
 * @link https://wiki.yandex-team.ru/videoplatform/ugc-live/lineandepisodehandles
 */
@AllArgsConstructor
public class UgcLiveClientImpl implements UgcLiveClient {

    private final HttpClient httpClient;
    private final URI baseUrl;
    private final String channelId;
    private final UserTicketSupplier userTicketSupplier;

    private final ObjectMapper objectMapper = new ObjectMapper()
            .registerModule(new BoltsModule())
            .registerModule(new StringEnumModule())
            .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);

    private final JacksonResponseHandlerFactory responseHandlerFactory =
            new JacksonResponseHandlerFactory(objectMapper);
    private final ResponseHandler<JsonNode> jsonNodeResponseHandler =
            responseHandlerFactory.cons(JsonNode.class);
    private final ResponseHandler<Option<JsonNode>> optionalJsonResponseHandler =
            responseHandlerFactory.option(JsonNode.class);

    @Override
    public long createLine(String name) {
        URI uri = new UriBuilder(baseUrl)
                .appendPath("line")
                .appendPath(channelId)
                .build();

        HttpPost request = createPostRequest(uri, Cf.map("name", TextNode.valueOf(name)));
        JsonNode node = ApacheHttpClientUtils.execute(request, httpClient, jsonNodeResponseHandler);

        return node.get("line_id").asLong();
    }

    @Override
    public LineInfo getLineInfo(long lineId) {
        URI uri = new UriBuilder(baseUrl)
                .appendPath("line")
                .appendPath(channelId)
                .appendPath(Long.toString(lineId))
                .build();

        return ApacheHttpClientUtils.execute(
                createGetRequest(uri), httpClient, responseHandlerFactory.cons(LineInfo.class));
    }

    @Override
    public void deleteLine(long lineId) {
        executeDeleteRequest(new UriBuilder(baseUrl)
                .appendPath("line")
                .appendPath(channelId)
                .appendPath(Long.toString(lineId))
                .build());
    }

    @Override
    public String createEpisode(long lineId, String title) {
        URI uri = new UriBuilder(baseUrl)
                .appendPath("episode")
                .appendPath(channelId)
                .build();

        HttpPost request = createPostRequest(uri, Cf.map(
                "line_id", LongNode.valueOf(lineId),
                "title", TextNode.valueOf(title)));
        JsonNode node = ApacheHttpClientUtils.execute(request, httpClient, jsonNodeResponseHandler);

        return node.get("slug").asText();
    }

    @Override
    public Option<StreamState> getStreamState(String episodeSlug) {
        URI uri = new UriBuilder(baseUrl)
                .appendPath("episode")
                .appendPath(channelId)
                .appendPath(episodeSlug)
                .build();

        HttpGet request = createGetRequest(uri);

        Option<JsonNode> node = ApacheHttpClientUtils.execute(request, httpClient, optionalJsonResponseHandler);

        return node.map(n -> StreamState.R.fromValue(n.get("stream_state").asText()));
    }

    @Override
    public void performStreamAction(String episodeSlug, StreamAction action) {
        URI uri = new UriBuilder(baseUrl)
                .appendPath("episode")
                .appendPath(channelId)
                .appendPath(episodeSlug)
                .appendPath("action")
                .build();

        HttpPost request = createPostRequest(uri, Cf.map("action", TextNode.valueOf(action.value())));
        ApacheHttpClientUtils.execute(request, httpClient, jsonNodeResponseHandler);
    }

    @Override
    public void deleteEpisode(String episodeSlug) {
        executeDeleteRequest(new UriBuilder(baseUrl)
                .appendPath("episode")
                .appendPath(channelId)
                .appendPath(episodeSlug)
                .build());
    }

    @Override
    public StreamSchedule getSchedule() {
        URI uri = new UriBuilder(baseUrl)
                .appendPath("schedule")
                .appendPath(channelId)
                .build();

        return ApacheHttpClientUtils.execute(
                createGetRequest(uri), httpClient, responseHandlerFactory.cons(StreamSchedule.class));
    }

    private HttpGet createGetRequest(URI uri) {
        HttpGet request = new HttpGet(uri);
        addHeaders(request);

        return request;
    }

    private HttpPost createPostRequest(URI uri, MapF<String, JsonNode> json) {
        HttpPost request = new HttpPost(uri);
        addHeaders(request);
        setBody(request, json);

        return request;
    }

    private HttpDelete createDeleteRequest(URI uri) {
        HttpDelete request = new HttpDelete(uri);
        addHeaders(request);

        return request;
    }

    private void addHeaders(HttpUriRequest request) {
        request.addHeader(TvmHeaders.USER_TICKET, userTicketSupplier.getUserTicket());
        RequestIdStack.current()
                .map(rid -> rid + "_" + Random2.R.nextAlnum(3))
                .forEach(rid -> request.addHeader("X-Req-Id", rid));
    }

    private void setBody(HttpEntityEnclosingRequest request, MapF<String, JsonNode> json) {
        setBody(request, new ObjectNode(objectMapper.getNodeFactory(), json));
    }

    private void setBody(HttpEntityEnclosingRequest request, JsonNode json) {
        byte[] serialized;
        try {
            serialized = objectMapper.writeValueAsBytes(json);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
        request.setEntity(new ByteArrayEntity(serialized, ContentType.APPLICATION_JSON));
    }

    private void executeDeleteRequest(URI uri) {
        HttpDelete request = createDeleteRequest(uri);

        ApacheHttpClientUtils.execute(request, httpClient, new ReadOptionalStringResponseHandler());
    }
}
