package ru.yandex.chemodan.app.telemost.ugcLive;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import lombok.AllArgsConstructor;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.ugcLive.model.StreamAction;
import ru.yandex.chemodan.app.telemost.ugcLive.model.StreamState;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.factory.ThreadNameThreadFactory;

public class UgcLiveStreamPublisherImpl implements UgcLiveStreamPublisher {

    private static final Logger logger = LoggerFactory.getLogger(UgcLiveStreamPublisherImpl.class);

    private final UgcLiveClient ugcLiveClient;
    private final DynamicProperty<ReschedulePolicy> reschedulePolicy;
    private final ScheduledExecutorService scheduler;

    public UgcLiveStreamPublisherImpl(UgcLiveClient ugcLiveClient) {
        this(ugcLiveClient, Duration.standardSeconds(10));
    }

    public UgcLiveStreamPublisherImpl(
            UgcLiveClient ugcLiveClient,
            Duration defaultRescheduleDelay
    ) {
        this.ugcLiveClient = ugcLiveClient;
        this.reschedulePolicy = new DynamicProperty<>(
                "telemost-ugc-live-stream-publish-reschedule",
                new RescheduleConstant(defaultRescheduleDelay, 300)
        );
        this.scheduler = Executors.newSingleThreadScheduledExecutor(
                new ThreadNameThreadFactory(UgcLiveStreamPublisherImpl.class)
        );
    }

    @Override
    public void submitPublishing(String episodeSlug) {
        schedule(episodeSlug, 1);
    }

    private void run(PublishRunnable runnable) {
        if (!publishIfReadySafe(runnable.slug)) {
            schedule(runnable.slug, runnable.attempt);
        }
    }

    private boolean publishIfReadySafe(String slug) {
        try {
            return publishIfReady(slug);

        } catch (Throwable e) {
            logger.warn("Failed to publish {}: {}", slug, e);
            return false;
        }
    }

    private boolean publishIfReady(String slug) {
        Option<StreamState> state = ugcLiveClient.getStreamState(slug);

        logger.info("Stream {} state now is {}", slug, state.orElse((StreamState) null));

        if (state.isSome(StreamState.READY)) {
            ugcLiveClient.performStreamAction(slug, StreamAction.PUBLISH);

            logger.info("Stream {} published successfully", slug);
        }
        return !state.isSome(StreamState.OFFLINE) && !state.isSome(StreamState.PREPARING);
    }

    private void schedule(String slug, int attempt) {
        Instant now = Instant.now();
        Option<Instant> scheduleTime = reschedulePolicy.get().rescheduleAt(now, attempt);

        if (scheduleTime.isPresent()) {
            logger.info("Schedule {} publishing to be executed at {}", slug, scheduleTime.get());

            scheduler.schedule(
                    new PublishRunnable(slug, attempt),
                    new Duration(now, scheduleTime.get()).getMillis(),
                    TimeUnit.MILLISECONDS);
        } else {
            logger.info("Wont schedule {} publishing", slug);
        }
    }

    @AllArgsConstructor
    private class PublishRunnable implements Runnable {
        private final String slug;
        private final int attempt;

        @Override
        public void run() {
            UgcLiveStreamPublisherImpl.this.run(this);
        }
    }
}
