package ru.yandex.chemodan.app.telemost.web.v1.actions;

import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.services.ClientConfigurationService;
import ru.yandex.chemodan.app.telemost.services.ConferencePeerService;
import ru.yandex.chemodan.app.telemost.services.ConferenceService;
import ru.yandex.chemodan.app.telemost.services.LimitsService;
import ru.yandex.chemodan.app.telemost.services.OverloadService;
import ru.yandex.chemodan.app.telemost.services.RoomService;
import ru.yandex.chemodan.app.telemost.services.UserService;
import ru.yandex.chemodan.app.telemost.services.model.Conference;
import ru.yandex.chemodan.app.telemost.services.model.ConferenceClientParameters;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.chemodan.app.telemost.services.model.User;
import ru.yandex.chemodan.app.telemost.web.v1.model.ConferenceData;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestHeader;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.tvm2.TvmHeaders;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.io.http.HttpStatus;

@ActionContainer
public class ConferencesActions {

    private final ConferenceService conferenceService;
    private final RoomService roomService;
    private final ClientConfigurationService clientConfigurationService;
    private final ConferencePeerService conferencePeerService;
    private final LimitsService limitsService;
    private final UserService userService;
    private final OverloadService overloadService;

    private final DynamicProperty<Boolean> enableMediatorRequests = new DynamicProperty<Boolean>("telemost-v1-mediator-enabled", false);

    public ConferencesActions(ConferenceService conferenceService, RoomService roomService,
            ClientConfigurationService clientConfigurationService, ConferencePeerService conferencePeerService,
            LimitsService limitsService, UserService userService, OverloadService overloadService)
    {
        this.conferenceService = conferenceService;
        this.roomService = roomService;
        this.clientConfigurationService = clientConfigurationService;
        this.conferencePeerService = conferencePeerService;
        this.limitsService = limitsService;
        this.userService = userService;
        this.overloadService = overloadService;
    }

    private void addUserIfNotExists(Option<PassportOrYaTeamUid> uid) {
        uid.ifPresent(userService::addUserIfNotExists);
    }

    @Path(value = "/v1/conferences", methods = {HttpMethod.POST})
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public ConferenceData createConference(@RequestParam("uid") Option<PassportOrYaTeamUid> uid,
                                           @RequestParam("staffOnly") Option<Boolean> isStaffOnly,
                                           @RequestParam("isPermanent") Option<Boolean> isPermanent,
                                           @RequestParam("externalMeeting") Option<Boolean> isExternalMeeting,
                                           @SpecialParam ActionInvocationContext invocationContext)
    {
        addUserIfNotExists(uid);
        overloadService.checkOverload();

        Option<User> user = uid.map(conferencePeerService::findUser);
        Conference conference = conferenceService.generateConference(
                ConferenceClientParameters.builder()
                        .user(user)
                        .staffOnly(isStaffOnly)
                        .permanent(isPermanent)
                        .externalMeeting(isExternalMeeting)
                        .eventId(Option.empty()).build()
        );
        if (enableMediatorRequests.get()) {
            roomService.createRoom(conference);
        }
        MapF<String, Object> clientConfiguration = clientConfigurationService.getV1ClientConfiguration();
        invocationContext.getHttpContext().setStatusCode(HttpStatus.SC_201_CREATED);
        return new ConferenceData(conference.withLimitType(limitsService.getV1LimitType(conference.getLimitType())),
                clientConfiguration);
    }

    @Path(value = "/v1/conferences/{uri:.*}/connection")
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public ConferenceData getConference(@PathParam("uri") String uri,
                                        @RequestParam("uid") Option<PassportOrYaTeamUid> uid,
                                        @RequestHeader(value = TvmHeaders.USER_TICKET, required = false) Option<String> tvmUserTicket)
    {
        addUserIfNotExists(uid);
        overloadService.checkOverload(uri, uid);

        Option<User> user = uid.map(conferencePeerService::findUser);
        Conference conference = conferenceService.joinConference(user, uri, tvmUserTicket, Option.empty());
        if (enableMediatorRequests.get()) {
            roomService.createRoom(conference);
        }
        return new ConferenceData(conference.withLimitType(limitsService.getV1LimitType(conference.getLimitType())),
                clientConfigurationService.getV1ClientConfiguration());
    }
}
