package ru.yandex.chemodan.app.telemost.web.v2.actions;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.exceptions.ForbiddenAccessToCreateBroadcast;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceDto;
import ru.yandex.chemodan.app.telemost.services.BroadcastService;
import ru.yandex.chemodan.app.telemost.services.ChatService;
import ru.yandex.chemodan.app.telemost.services.ConferenceService;
import ru.yandex.chemodan.app.telemost.services.StreamService;
import ru.yandex.chemodan.app.telemost.services.UserService;
import ru.yandex.chemodan.app.telemost.services.model.Broadcast;
import ru.yandex.chemodan.app.telemost.services.model.BroadcastAndConferenceUris;
import ru.yandex.chemodan.app.telemost.services.model.BroadcastUserId;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.chemodan.app.telemost.services.model.Stream;
import ru.yandex.chemodan.app.telemost.web.v2.model.BroadcastData;
import ru.yandex.chemodan.app.telemost.web.v2.model.BroadcastInitData;
import ru.yandex.chemodan.app.telemost.web.v2.model.StreamConnectionData;
import ru.yandex.chemodan.app.telemost.web.v2.model.StreamData;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.BoundByJackson;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestHeader;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.inside.passport.tvm2.TvmHeaders;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@ActionContainer
@AllArgsConstructor
public class BroadcastsActionsV2 {
    private static final Logger logger = LoggerFactory.getLogger(BroadcastsActionsV2.class);

    private final ConferenceService conferenceService;

    private final BroadcastService broadcastService;

    private final StreamService streamService;

    private final ChatService chatService;

    private final UserService userService;

    @Path(value = "/v2/conferences/{uri:.*}/broadcast", methods = {HttpMethod.PUT})
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public BroadcastData createBroadcast(@PathParam("uri") String uri,
                                         @RequestParam(value = "uid") PassportOrYaTeamUid uid,
                                         @RequestHeader(value = TvmHeaders.USER_TICKET, required = false) Option<String> tvmUserTicket,
                                         @BoundByJackson BroadcastInitData broadcastInitData)
    {
        ConferenceDto conferenceDto = conferenceService.getConferenceByFullUri(uri);
        conferenceService.ensureIsAdmin(conferenceDto, uid);
        if (!conferenceService.isBroadcastFeatureEnabled(uid)) {
            throw new ForbiddenAccessToCreateBroadcast();
        }

        // TODO: Временное решение на пилот. Тексты будем брать из танкера
        if (broadcastInitData.getCaption().orElse("").isEmpty()
                && broadcastInitData.getDescription().orElse("").isEmpty())
        {
            broadcastInitData = new BroadcastInitData(Option.of("Без названия"), Option.empty());
        }

        Broadcast broadcast = broadcastService.createBroadcast(conferenceDto.getId(), uid.asString(),
                broadcastInitData);

        // Создаем чат при создании трансляции в следующих случаях:
        // - если конференция не подвязана к событию календаря
        // - или если событие календаря есть и до его начала осталось менее часа
        return chatService.tryCreateBroadcastChatIfNeed(conferenceDto, Option.of(uid), tvmUserTicket)
                .map(x -> new BroadcastData(Broadcast.withChatPath(broadcast, Option.of(x.getChatPath()))))
                .orElse(new BroadcastData(broadcast));
    }

    @Path(value = "/v2/conferences/{uri:.*}/broadcast/{broadcast_uri:.*}/start", methods = {HttpMethod.POST})
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public StreamData startBroadcast(@PathParam("uri") String uri,
                                     @PathParam("broadcast_uri") String broadcastUri,
                                     @RequestParam(value = "uid") PassportOrYaTeamUid uid)
    {

        conferenceService.checkAdmin(conferenceService.findConferenceByUriId(
                conferenceService.getConferenceUriId(uri).getShortUrlId()).get(), uid
        );
        BroadcastAndConferenceUris uris = new BroadcastAndConferenceUris(broadcastUri, uri);
        return createStreamData(streamService.startStream(uid, uris));
    }

    @Path(value = "/v2/conferences/{uri:.*}/broadcast/{broadcast_uri:.*}/stop", methods = {HttpMethod.POST})
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public StreamData stopBroadcast(@PathParam("uri") String uri,
                                    @PathParam("broadcast_uri") String broadcastUri,
                                    @RequestParam("uid") Option<PassportOrYaTeamUid> uid,
                                    @RequestParam("translatorToken") Option<String> translatorToken)
    {
        BroadcastUserId userId = new BroadcastUserId(uid, translatorToken);
        BroadcastAndConferenceUris uris = new BroadcastAndConferenceUris(broadcastUri, uri);
        return createStreamData(streamService.stopStream(userId, uris));
    }

    @Path(value = "/v2/broadcast/{broadcast_uri:.*}/connection", methods = {HttpMethod.GET})
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public StreamConnectionData getConnectionData(@PathParam("broadcast_uri") String broadcastUri)
    {
        chatService.tryCreateBroadcastChatIfNeed(broadcastUri);
        return new StreamConnectionData(streamService.getConnection(broadcastUri));
    }

    private StreamData createStreamData(Stream stream) {
        return new StreamData(
                stream.getStream().getOwnerUid().get(),
                stream.getStream().getStartedAt().get().getMillis() / 1000,
                stream.getStream().getStoppedAt().map(t -> t.getMillis() / 1000),
                stream.getStream().getStoppedAt().isEmpty() ? Option.of(streamService.getStreamUri(stream.getStream().getUgcLiveSlug())) : Option.empty());
    }
}
